<?php
require_once '../config/config.php';
requireLogin('company');

$database = new Database();
$db = $database->getConnection();
$company_id = $_SESSION['company_id'];

// Handle approval/rejection/deactivation actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $candidate_id = (int)$_POST['candidate_id'];
    $action = $_POST['action'];
    $current_user_id = $_SESSION['user_id']; // Company user ID
    
    if ($action === 'approve') {
        // Get candidate details before updating
        $stmt = $db->prepare("SELECT * FROM candidates WHERE id = ? AND company_id = ?");
        $stmt->execute([$candidate_id, $company_id]);
        $candidate = $stmt->fetch();
        
        if ($candidate) {
            // Approve internal candidate
            $stmt = $db->prepare("UPDATE candidates SET approval_status = 'approved', approved_at = NOW(), approved_by = ? WHERE id = ? AND company_id = ?");
            $stmt->execute([$company_id, $candidate_id, $company_id]);
            
            // Send email notification to approved candidate
            require_once '../includes/notification_system_smtp.php';
            $notificationSystem = new NotificationSystemSMTP($db);
            $notificationSystem->notifyInternalCandidateApproval($candidate_id, 'approved');
            
            $message = "Internal candidate approved successfully. They can now login to their account and have been notified via email.";
            $message_type = "success";
        }
    } elseif ($action === 'reject') {
        // Get candidate details before updating
        $stmt = $db->prepare("SELECT * FROM candidates WHERE id = ? AND company_id = ?");
        $stmt->execute([$candidate_id, $company_id]);
        $candidate = $stmt->fetch();
        
        if ($candidate) {
            $rejection_reason = $_POST['rejection_reason'] ?? '';
            $stmt = $db->prepare("UPDATE candidates SET approval_status = 'rejected', rejection_reason = ? WHERE id = ? AND company_id = ?");
            $stmt->execute([$rejection_reason, $candidate_id, $company_id]);
            
            // Send email notification to rejected candidate
            require_once '../includes/notification_system_smtp.php';
            $notificationSystem = new NotificationSystemSMTP($db);
            $notificationSystem->notifyInternalCandidateApproval($candidate_id, 'rejected', $rejection_reason);
            
            $message = "Internal candidate rejected and has been notified via email.";
            $message_type = "info";
        }
    } elseif ($action === 'deactivate') {
        // Get candidate details before deactivating
        $stmt = $db->prepare("SELECT * FROM candidates WHERE id = ? AND company_id = ?");
        $stmt->execute([$candidate_id, $company_id]);
        $candidate = $stmt->fetch();
        
        if ($candidate) {
            $deactivation_reason = $_POST['deactivation_reason'] ?? '';
            $deactivation_type = $_POST['deactivation_type'] ?? 'inactive'; // inactive or terminated
            
            // Update candidate status
            $stmt = $db->prepare("
                UPDATE candidates 
                SET employment_status = ?, 
                    deactivated_at = NOW(), 
                    deactivated_by = ?,
                    deactivation_reason = ?
                WHERE id = ? AND company_id = ?
            ");
            $stmt->execute([$deactivation_type, $current_user_id, $deactivation_reason, $candidate_id, $company_id]);
            
            // Send notification email to deactivated candidate
            require_once '../includes/notification_system_smtp.php';
            $notificationSystem = new NotificationSystemSMTP($db);
            $notificationSystem->notifyInternalCandidateDeactivation($candidate_id, $deactivation_type, $deactivation_reason);
            
            $action_text = $deactivation_type === 'terminated' ? 'terminated' : 'deactivated';
            $message = "Internal candidate {$action_text} successfully. They will no longer see internal job advertisements and have been notified via email.";
            $message_type = "warning";
        }
    } elseif ($action === 'reactivate') {
        // Reactivate a previously deactivated candidate
        $stmt = $db->prepare("SELECT * FROM candidates WHERE id = ? AND company_id = ?");
        $stmt->execute([$candidate_id, $company_id]);
        $candidate = $stmt->fetch();
        
        if ($candidate) {
            $stmt = $db->prepare("
                UPDATE candidates 
                SET employment_status = 'active', 
                    deactivated_at = NULL, 
                    deactivated_by = NULL,
                    deactivation_reason = NULL
                WHERE id = ? AND company_id = ?
            ");
            $stmt->execute([$candidate_id, $company_id]);
            
            $message = "Internal candidate reactivated successfully. They can now see internal job advertisements again.";
            $message_type = "success";
        }
    }
}

// Get internal candidates for this company
$stmt = $db->prepare("
    SELECT c.*, c.created_at as application_date, c.approval_status,
           c.employee_id, c.current_department, c.current_position,
           c.rejection_reason, c.approved_at,
           c.employment_status, c.deactivated_at, c.deactivation_reason,
           cu.name as deactivated_by_name
    FROM candidates c
    LEFT JOIN company_users cu ON c.deactivated_by = cu.id
    WHERE c.company_id = ? AND c.candidate_type = 'internal'
    ORDER BY c.employment_status ASC, c.created_at DESC
");
$stmt->execute([$company_id]);
$internal_candidates = $stmt->fetchAll();

// Get statistics
$stats = [];
$stmt = $db->prepare("SELECT approval_status, employment_status, COUNT(*) as count FROM candidates WHERE company_id = ? AND candidate_type = 'internal' GROUP BY approval_status, employment_status");
$stmt->execute([$company_id]);
$all_counts = $stmt->fetchAll();

// Initialize stats
$stats['pending'] = 0;
$stats['approved'] = 0;
$stats['rejected'] = 0;
$stats['active'] = 0;
$stats['inactive'] = 0;
$stats['terminated'] = 0;
$stats['total'] = 0;

// Calculate stats
foreach ($all_counts as $count_row) {
    $stats[$count_row['approval_status']] += $count_row['count'];
    $stats[$count_row['employment_status']] += $count_row['count'];
    $stats['total'] += $count_row['count'];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Internal Candidates - <?php echo SITE_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container-fluid">
            <a class="navbar-brand" href="../index.php">
                <i class="fas fa-briefcase me-2"></i><?php echo SITE_NAME; ?>
            </a>
            
            <div class="navbar-nav ms-auto">
                <span class="navbar-text me-3">
                    Welcome, <?php echo htmlspecialchars($_SESSION['company_name']); ?>
                </span>
                <a class="nav-link" href="../auth/logout.php">
                    <i class="fas fa-sign-out-alt me-1"></i>Logout
                </a>
            </div>
        </div>
    </nav>

    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <nav class="col-md-3 col-lg-2 sidebar">
                <div class="position-sticky">
                    <ul class="nav flex-column">
                        <li class="nav-item">
                            <a class="nav-link" href="dashboard.php">
                                <i class="fas fa-tachometer-alt me-2"></i>Dashboard
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="profile.php">
                                <i class="fas fa-building me-2"></i>Company Profile
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="jobs.php">
                                <i class="fas fa-briefcase me-2"></i>Manage Jobs
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="post-job.php">
                                <i class="fas fa-plus-circle me-2"></i>Post New Job
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="applications.php">
                                <i class="fas fa-file-alt me-2"></i>Applications
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link active" href="internal-candidates.php">
                                <i class="fas fa-user-tie me-2"></i>Internal Candidates
                                <?php if ($stats['pending'] > 0): ?>
                                    <span class="badge bg-warning ms-2"><?php echo $stats['pending']; ?></span>
                                <?php endif; ?>
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="interviews.php">
                                <i class="fas fa-calendar-alt me-2"></i>Interviews
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="hr-analytics.php">
                                <i class="fas fa-chart-line me-2"></i>HR Analytics
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="change-password.php">
                                <i class="fas fa-lock me-2"></i>Change Password
                            </a>
                        </li>
                    </ul>
                </div>
            </nav>

            <!-- Main Content -->
            <main class="col-md-9 col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-user-tie me-2"></i>Internal Candidates</h1>
                </div>

                <?php if (isset($message)): ?>
                    <div class="alert alert-<?php echo $message_type; ?> alert-dismissible fade show" role="alert">
                        <?php echo htmlspecialchars($message); ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <!-- Statistics Cards -->
                <div class="row mb-4">
                    <div class="col-md-2">
                        <div class="card dashboard-card text-center">
                            <div class="card-body">
                                <h3 class="text-warning"><?php echo $stats['pending']; ?></h3>
                                <p class="card-text">Pending</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="card dashboard-card text-center">
                            <div class="card-body">
                                <h3 class="text-success"><?php echo $stats['active']; ?></h3>
                                <p class="card-text">Active</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="card dashboard-card text-center">
                            <div class="card-body">
                                <h3 class="text-info"><?php echo $stats['inactive']; ?></h3>
                                <p class="card-text">Inactive</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="card dashboard-card text-center">
                            <div class="card-body">
                                <h3 class="text-danger"><?php echo $stats['terminated']; ?></h3>
                                <p class="card-text">Terminated</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="card dashboard-card text-center">
                            <div class="card-body">
                                <h3 class="text-secondary"><?php echo $stats['rejected']; ?></h3>
                                <p class="card-text">Rejected</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="card dashboard-card text-center">
                            <div class="card-body">
                                <h3 class="text-primary"><?php echo $stats['total']; ?></h3>
                                <p class="card-text">Total</p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Internal Candidates Table -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Internal Transfer Requests</h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($internal_candidates)): ?>
                            <div class="text-center py-4">
                                <i class="fas fa-user-tie fa-3x text-muted mb-3"></i>
                                <p class="text-muted">No internal candidate requests found.</p>
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Employee</th>
                                            <th>Current Position</th>
                                            <th>Applied Date</th>
                                            <th>Approval Status</th>
                                            <th>Employment Status</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($internal_candidates as $candidate): ?>
                                            <tr>
                                                <td>
                                                    <div class="d-flex align-items-center">
                                                        <?php if ($candidate['profile_picture']): ?>
                                                            <img src="<?php echo UPLOAD_PATH; ?>profiles/<?php echo htmlspecialchars($candidate['profile_picture']); ?>" 
                                                                 class="rounded-circle me-2" width="40" height="40" alt="Profile">
                                                        <?php else: ?>
                                                            <div class="bg-secondary rounded-circle d-flex align-items-center justify-content-center me-2" 
                                                                 style="width: 40px; height: 40px;">
                                                                <i class="fas fa-user text-white"></i>
                                                            </div>
                                                        <?php endif; ?>
                                                        <div>
                                                            <strong><?php echo htmlspecialchars($candidate['first_name'] . ' ' . $candidate['last_name']); ?></strong>
                                                            <br><small class="text-muted">ID: <?php echo htmlspecialchars($candidate['employee_id']); ?></small>
                                                        </div>
                                                    </div>
                                                </td>
                                                <td>
                                                    <strong><?php echo htmlspecialchars($candidate['current_position']); ?></strong>
                                                    <br><small class="text-muted"><?php echo htmlspecialchars($candidate['current_department']); ?></small>
                                                </td>
                                                <td><?php echo date('M d, Y', strtotime($candidate['application_date'])); ?></td>
                                                <td>
                                                    <?php
                                                    $approvalStatusClass = '';
                                                    switch ($candidate['approval_status']) {
                                                        case 'pending': $approvalStatusClass = 'bg-warning'; break;
                                                        case 'approved': $approvalStatusClass = 'bg-success'; break;
                                                        case 'rejected': $approvalStatusClass = 'bg-danger'; break;
                                                    }
                                                    ?>
                                                    <span class="badge <?php echo $approvalStatusClass; ?>"><?php echo ucfirst($candidate['approval_status']); ?></span>
                                                </td>
                                                <td>
                                                    <?php
                                                    $employmentStatusClass = '';
                                                    switch ($candidate['employment_status']) {
                                                        case 'active': $employmentStatusClass = 'bg-success'; break;
                                                        case 'inactive': $employmentStatusClass = 'bg-secondary'; break;
                                                        case 'terminated': $employmentStatusClass = 'bg-danger'; break;
                                                    }
                                                    ?>
                                                    <span class="badge <?php echo $employmentStatusClass; ?>"><?php echo ucfirst($candidate['employment_status']); ?></span>
                                                    <?php if ($candidate['employment_status'] !== 'active' && $candidate['deactivated_at']): ?>
                                                        <br><small class="text-muted">
                                                            <?php echo date('M d, Y', strtotime($candidate['deactivated_at'])); ?>
                                                            <?php if ($candidate['deactivated_by_name']): ?>
                                                                by <?php echo htmlspecialchars($candidate['deactivated_by_name']); ?>
                                                            <?php endif; ?>
                                                        </small>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?php if ($candidate['approval_status'] === 'pending'): ?>
                                                        <!-- Pending approval actions -->
                                                        <div class="btn-group" role="group">
                                                            <button type="button" class="btn btn-sm btn-success" 
                                                                    onclick="approveCandidate(<?php echo $candidate['id']; ?>)">
                                                                <i class="fas fa-check"></i> Approve
                                                            </button>
                                                            <button type="button" class="btn btn-sm btn-danger" 
                                                                    onclick="showRejectModal(<?php echo $candidate['id']; ?>, '<?php echo addslashes($candidate['first_name'] . ' ' . $candidate['last_name']); ?>')">
                                                                <i class="fas fa-times"></i> Reject
                                                            </button>
                                                        </div>
                                                    <?php elseif ($candidate['employment_status'] === 'active'): ?>
                                                        <!-- Active employee actions -->
                                                        <div class="btn-group" role="group">
                                                            <button type="button" class="btn btn-sm btn-outline-primary"
                                                                    onclick="viewCandidate(<?php echo $candidate['id']; ?>)">
                                                                <i class="fas fa-eye"></i> View
                                                            </button>
                                                            <button type="button" class="btn btn-sm btn-warning" 
                                                                    onclick="showDeactivateModal(<?php echo $candidate['id']; ?>, '<?php echo addslashes($candidate['first_name'] . ' ' . $candidate['last_name']); ?>', 'inactive')">
                                                                <i class="fas fa-pause"></i> Deactivate
                                                            </button>
                                                            <button type="button" class="btn btn-sm btn-danger" 
                                                                    onclick="showDeactivateModal(<?php echo $candidate['id']; ?>, '<?php echo addslashes($candidate['first_name'] . ' ' . $candidate['last_name']); ?>', 'terminated')">
                                                                <i class="fas fa-user-times"></i> Terminate
                                                            </button>
                                                        </div>
                                                    <?php elseif ($candidate['employment_status'] === 'inactive'): ?>
                                                        <!-- Inactive employee actions -->
                                                        <div class="btn-group" role="group">
                                                            <button type="button" class="btn btn-sm btn-outline-primary"
                                                                    onclick="viewCandidate(<?php echo $candidate['id']; ?>)">
                                                                <i class="fas fa-eye"></i> View
                                                            </button>
                                                            <button type="button" class="btn btn-sm btn-success" 
                                                                    onclick="reactivateCandidate(<?php echo $candidate['id']; ?>)">
                                                                <i class="fas fa-play"></i> Reactivate
                                                            </button>
                                                            <button type="button" class="btn btn-sm btn-danger" 
                                                                    onclick="showDeactivateModal(<?php echo $candidate['id']; ?>, '<?php echo addslashes($candidate['first_name'] . ' ' . $candidate['last_name']); ?>', 'terminated')">
                                                                <i class="fas fa-user-times"></i> Terminate
                                                            </button>
                                                        </div>
                                                    <?php else: ?>
                                                        <!-- Terminated employee actions -->
                                                        <button type="button" class="btn btn-sm btn-outline-primary"
                                                                onclick="viewCandidate(<?php echo $candidate['id']; ?>)">
                                                            <i class="fas fa-eye"></i> View Details
                                                        </button>
                                                        <?php if ($candidate['deactivation_reason']): ?>
                                                            <button type="button" class="btn btn-sm btn-outline-info"
                                                                    onclick="showDeactivationDetails('<?php echo addslashes($candidate['deactivation_reason']); ?>', '<?php echo date('M d, Y', strtotime($candidate['deactivated_at'])); ?>', '<?php echo htmlspecialchars($candidate['deactivated_by_name'] ?? 'System'); ?>')">
                                                                <i class="fas fa-info-circle"></i> Details
                                                            </button>
                                                        <?php endif; ?>
                                                    <?php endif; ?>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <!-- Rejection Modal -->
    <div class="modal fade" id="rejectModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Reject Internal Candidate</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="action" value="reject">
                        <input type="hidden" name="candidate_id" id="rejectCandidateId">
                        
                        <p>Are you sure you want to reject <strong id="candidateName"></strong>?</p>
                        
                        <div class="mb-3">
                            <label for="rejection_reason" class="form-label">Reason for Rejection (Optional)</label>
                            <textarea class="form-control" name="rejection_reason" id="rejection_reason" rows="3" 
                                      placeholder="Provide a reason for the rejection..."></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-danger">Reject Candidate</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Deactivate/Terminate Modal -->
    <div class="modal fade" id="deactivateModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="deactivateModalTitle">Deactivate Internal Candidate</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="action" value="deactivate">
                        <input type="hidden" name="candidate_id" id="deactivateCandidateId">
                        <input type="hidden" name="deactivation_type" id="deactivationType">
                        
                        <div class="alert alert-warning">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            <strong>Important:</strong> <span id="deactivateWarningText"></span>
                        </div>
                        
                        <p>Are you sure you want to <span id="deactivateActionText"></span> <strong id="deactivateCandidateName"></strong>?</p>
                        
                        <div class="mb-3">
                            <label for="deactivation_reason" class="form-label">Reason <span class="text-danger">*</span></label>
                            <textarea class="form-control" name="deactivation_reason" id="deactivation_reason" rows="3" 
                                      placeholder="Provide a detailed reason for this action..." required></textarea>
                            <small class="form-text text-muted">This reason will be emailed to the employee and logged for audit purposes.</small>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn" id="deactivateSubmitBtn">Confirm Action</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Deactivation Details Modal -->
    <div class="modal fade" id="deactivationDetailsModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Deactivation Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="row">
                        <div class="col-sm-4"><strong>Deactivated On:</strong></div>
                        <div class="col-sm-8" id="deactivationDate">-</div>
                    </div>
                    <div class="row mt-2">
                        <div class="col-sm-4"><strong>Deactivated By:</strong></div>
                        <div class="col-sm-8" id="deactivatedBy">-</div>
                    </div>
                    <div class="row mt-2">
                        <div class="col-sm-4"><strong>Reason:</strong></div>
                        <div class="col-sm-8" id="deactivationReasonText">-</div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function approveCandidate(candidateId) {
            if (confirm('Are you sure you want to approve this internal candidate?')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = `
                    <input type="hidden" name="action" value="approve">
                    <input type="hidden" name="candidate_id" value="${candidateId}">
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }

        function showRejectModal(candidateId, candidateName) {
            document.getElementById('rejectCandidateId').value = candidateId;
            document.getElementById('candidateName').textContent = candidateName;
            new bootstrap.Modal(document.getElementById('rejectModal')).show();
        }

        function showDeactivateModal(candidateId, candidateName, type) {
            document.getElementById('deactivateCandidateId').value = candidateId;
            document.getElementById('deactivationType').value = type;
            document.getElementById('deactivateCandidateName').textContent = candidateName;
            
            if (type === 'terminated') {
                document.getElementById('deactivateModalTitle').textContent = 'Terminate Employee';
                document.getElementById('deactivateActionText').textContent = 'terminate';
                document.getElementById('deactivateWarningText').textContent = 'This employee will be permanently marked as terminated and will no longer have access to internal job postings.';
                document.getElementById('deactivateSubmitBtn').className = 'btn btn-danger';
                document.getElementById('deactivateSubmitBtn').textContent = 'Terminate Employee';
            } else {
                document.getElementById('deactivateModalTitle').textContent = 'Deactivate Employee';
                document.getElementById('deactivateActionText').textContent = 'deactivate';
                document.getElementById('deactivateWarningText').textContent = 'This employee will be temporarily deactivated and will not see internal job postings until reactivated.';
                document.getElementById('deactivateSubmitBtn').className = 'btn btn-warning';
                document.getElementById('deactivateSubmitBtn').textContent = 'Deactivate Employee';
            }
            
            // Clear previous reason
            document.getElementById('deactivation_reason').value = '';
            
            new bootstrap.Modal(document.getElementById('deactivateModal')).show();
        }

        function reactivateCandidate(candidateId) {
            if (confirm('Are you sure you want to reactivate this employee? They will be able to see internal job postings again.')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = `
                    <input type="hidden" name="action" value="reactivate">
                    <input type="hidden" name="candidate_id" value="${candidateId}">
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }

        function showDeactivationDetails(reason, date, deactivatedBy) {
            document.getElementById('deactivationDate').textContent = date;
            document.getElementById('deactivatedBy').textContent = deactivatedBy;
            document.getElementById('deactivationReasonText').textContent = reason;
            
            new bootstrap.Modal(document.getElementById('deactivationDetailsModal')).show();
        }

        function viewCandidate(candidateId) {
            // Redirect to candidate details or open modal
            window.location.href = `view-internal-candidate.php?id=${candidateId}`;
        }
    </script>
</body>
</html>