<?php
/**
 * Simple Email Service - Fallback for when PHPMailer fails
 * Uses basic PHP mail() function
 */
class SimpleEmail {
    private $db;
    
    public function __construct($database) {
        $this->db = $database;
    }
    
    public function sendSimpleEmail($to_email, $subject, $message) {
        try {
            // Log email attempt
            $stmt = $this->db->prepare("
                INSERT INTO email_notifications (to_email, subject, message) 
                VALUES (?, ?, ?)
            ");
            $stmt->execute([$to_email, $subject, $message]);
            $email_id = $this->db->lastInsertId();
            
            // Simple headers
            $headers = "From: " . SITE_NAME . " <noreply@localhost>\r\n";
            $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
            $headers .= "MIME-Version: 1.0\r\n";
            
            // Simple HTML template
            $html_message = "
            <html>
            <head><title>$subject</title></head>
            <body style='font-family: Arial, sans-serif; line-height: 1.6; color: #333;'>
                <div style='max-width: 600px; margin: 0 auto; padding: 20px;'>
                    <h2 style='color: #0066cc;'>" . SITE_NAME . "</h2>
                    <div style='background: #f9f9f9; padding: 20px; border-radius: 5px;'>
                        $message
                    </div>
                    <hr style='margin: 20px 0;'>
                    <p style='font-size: 12px; color: #666;'>
                        This is an automated message. Please do not reply.
                    </p>
                </div>
            </body>
            </html>";
            
            // Try to send email
            if (mail($to_email, $subject, $html_message, $headers)) {
                // Mark as sent
                $stmt = $this->db->prepare("
                    UPDATE email_notifications 
                    SET status = 'sent', sent_at = NOW() 
                    WHERE id = ?
                ");
                $stmt->execute([$email_id]);
                return true;
            } else {
                // Mark as failed
                $stmt = $this->db->prepare("
                    UPDATE email_notifications 
                    SET status = 'failed', error_message = 'PHP mail() function failed' 
                    WHERE id = ?
                ");
                $stmt->execute([$email_id]);
                return false;
            }
            
        } catch (Exception $e) {
            error_log("Simple email sending failed: " . $e->getMessage());
            return false;
        }
    }
    
    public function sendStatusChangeEmail($application, $message, $status) {
        $subject = $this->getSubject($status, $application['job_title']);
        $email_body = $this->formatMessage($application, $message, $status);
        
        return $this->sendSimpleEmail($application['email'], $subject, $email_body);
    }
    
    private function getSubject($status, $job_title) {
        switch ($status) {
            case 'shortlisted':
                return "You've Been Shortlisted - $job_title";
            case 'interviewed':
                return "Interview Update - $job_title";
            case 'hired':
                return "Congratulations! You've Been Hired - $job_title";
            case 'rejected':
                return "Application Update - $job_title";
            default:
                return "Application Status Update - $job_title";
        }
    }
    
    private function formatMessage($application, $message, $status) {
        $status_color = [
            'shortlisted' => '#28a745',
            'interviewed' => '#17a2b8', 
            'hired' => '#28a745',
            'rejected' => '#dc3545'
        ][$status] ?? '#007bff';
        
        return "
        <h3 style='color: $status_color;'>Application Status Update</h3>
        <p><strong>Job:</strong> " . htmlspecialchars($application['job_title']) . "</p>
        <p><strong>Company:</strong> " . htmlspecialchars($application['company_name']) . "</p>
        <p><strong>Dear " . htmlspecialchars($application['first_name']) . ",</strong></p>
        <p>" . htmlspecialchars($message) . "</p>
        <div style='text-align: center; margin: 20px 0;'>
            <span style='background: $status_color; color: white; padding: 8px 16px; border-radius: 4px; font-weight: bold;'>
                " . strtoupper($status) . "
            </span>
        </div>
        <p style='margin-top: 20px;'>
            <a href='" . SITE_URL . "/candidate/applications.php' 
               style='background: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 4px;'>
                View My Applications
            </a>
        </p>";
    }
}
?>