<?php
// Include database configuration
require_once __DIR__ . '/database.php';

// Include security utilities
require_once __DIR__ . '/../includes/file_handler.php';
require_once __DIR__ . '/../includes/security.php';
require_once __DIR__ . '/../includes/hr_analytics.php';
require_once __DIR__ . '/../includes/excel_exporter.php';
require_once __DIR__ . '/../includes/excel_generator.php';

// Initialize security manager and secure session
$security = SecurityManager::getInstance();
$security->initializeSecureSession();
$security->addSecurityHeaders();

// Site configuration
define('SITE_NAME', 'Recruitment Portal');
define('SITE_URL', 'http://www.sanpcref.co.za/recruit');
define('UPLOAD_PATH', __DIR__ . '/../uploads/');
define('UPLOAD_URL', SITE_URL . '/uploads/');

// Create uploads directory if it doesn't exist with secure permissions
if (!file_exists(UPLOAD_PATH)) {
    mkdir(UPLOAD_PATH, 0755, true);
    mkdir(UPLOAD_PATH . 'profiles/', 0755, true);
    mkdir(UPLOAD_PATH . 'logos/', 0755, true);
    mkdir(UPLOAD_PATH . 'resumes/', 0755, true);
    mkdir(UPLOAD_PATH . 'certificates/', 0755, true);
    mkdir(UPLOAD_PATH . 'documents/', 0755, true);
    
    // Add .htaccess to prevent direct script execution
    $htaccessContent = "Options -ExecCGI\n";
    $htaccessContent .= "AddHandler cgi-script .php .pl .py .jsp .asp .sh .cgi\n";
    $htaccessContent .= "<Files ~ \"\\.(php|pl|py|jsp|asp|sh|cgi)$\">\n";
    $htaccessContent .= "    Order allow,deny\n";
    $htaccessContent .= "    Deny from all\n";
    $htaccessContent .= "</Files>\n";
    
    file_put_contents(UPLOAD_PATH . '.htaccess', $htaccessContent);
}

// Helper functions
function sanitize($data) {
    return htmlspecialchars(strip_tags(trim($data)));
}

function redirect($url) {
    header("Location: " . $url);
    exit();
}

function isLoggedIn($user_type = null) {
    if ($user_type) {
        return isset($_SESSION['user_type']) && $_SESSION['user_type'] === $user_type;
    }
    return isset($_SESSION['user_id']) && isset($_SESSION['user_type']);
}

function requireLogin($user_type = null) {
    if (!isLoggedIn($user_type)) {
        redirect(SITE_URL . '/auth/login.php');
    }
}

function formatDate($date) {
    return date('M d, Y', strtotime($date));
}

function formatDateTime($datetime) {
    return date('M d, Y g:i A', strtotime($datetime));
}

function generateCSRFToken() {
    $security = SecurityManager::getInstance();
    return $security->generateSecureCSRFToken();
}

function validateCSRFToken($token) {
    $security = SecurityManager::getInstance();
    return $security->validateCSRFToken($token);
}

function validatePasswordComplexity($password) {
    $errors = [];
    
    // Minimum length
    if (strlen($password) < 8) {
        $errors[] = 'Password must be at least 8 characters long';
    }
    
    // Maximum length
    if (strlen($password) > 128) {
        $errors[] = 'Password must not exceed 128 characters';
    }
    
    // Must contain uppercase letter
    if (!preg_match('/[A-Z]/', $password)) {
        $errors[] = 'Password must contain at least one uppercase letter';
    }
    
    // Must contain lowercase letter
    if (!preg_match('/[a-z]/', $password)) {
        $errors[] = 'Password must contain at least one lowercase letter';
    }
    
    // Must contain digit
    if (!preg_match('/[0-9]/', $password)) {
        $errors[] = 'Password must contain at least one number';
    }
    
    // Must contain special character
    if (!preg_match('/[!@#$%^&*()_+\-=\[\]{};\':"\\|,.<>\/?]/', $password)) {
        $errors[] = 'Password must contain at least one special character (!@#$%^&*()_+-=[]{}|;:,.<>?)';
    }
    
    return $errors;
}

function isPasswordStrong($password) {
    return empty(validatePasswordComplexity($password));
}
?>