<?php
require_once '../config/config.php';
requireLogin('company');

$database = new Database();
$db = $database->getConnection();

$company_id = $_SESSION['company_id'];
$application_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$error = '';
$success = '';

if (!$application_id) {
    redirect('applications.php');
}

// Get application details
$stmt = $db->prepare("
    SELECT ja.*, j.title as job_title, j.description as job_description, j.requirements, j.location, j.deadline,
           c.first_name, c.last_name, c.email, c.phone, c.address, c.date_of_birth, 
           c.gender, c.profile_picture, c.created_at as candidate_since,
           r.title as resume_title, r.summary, r.skills
    FROM job_applications ja
    JOIN jobs j ON ja.job_id = j.id
    JOIN candidates c ON ja.candidate_id = c.id
    LEFT JOIN resumes r ON ja.resume_id = r.id
    WHERE ja.id = ? AND j.company_id = ?
");
$stmt->execute([$application_id, $company_id]);
$application = $stmt->fetch();

if (!$application) {
    redirect('applications.php');
}

$candidate_id = $application['candidate_id'];

// Check if shortlisting is allowed (only after deadline)
$shortlisting_allowed = true;
$deadline_message = '';
if ($application['deadline']) {
    $current_date = date('Y-m-d');
    if ($current_date <= $application['deadline']) {
        $shortlisting_allowed = false;
        $deadline_message = 'Shortlisting will be available after ' . formatDate($application['deadline']);
    }
}

// Get candidate's education
$stmt = $db->prepare("SELECT * FROM education WHERE candidate_id = ? ORDER BY start_date DESC");
$stmt->execute([$candidate_id]);
$education = $stmt->fetchAll();

// Get candidate's work experience
$stmt = $db->prepare("SELECT * FROM work_experience WHERE candidate_id = ? ORDER BY start_date DESC");
$stmt->execute([$candidate_id]);
$experience = $stmt->fetchAll();

// Get candidate's survey responses
$stmt = $db->prepare("
    SELECT sr.response, sq.question, sq.question_type
    FROM job_application_survey_responses sr
    JOIN job_survey_questions sq ON sr.question_id = sq.id
    WHERE sr.application_id = ?
    ORDER BY sq.question_order ASC
");
$stmt->execute([$application_id]);
$survey_responses = $stmt->fetchAll();

// Get candidate's documents (attachments)
$stmt = $db->prepare("
    SELECT id, document_type, title, file_path, file_size, file_type, description, created_at
    FROM candidate_documents 
    WHERE candidate_id = ? AND is_public = 1
    ORDER BY document_type, created_at DESC
");
$stmt->execute([$candidate_id]);
$candidate_documents = $stmt->fetchAll();

// Get candidate's certificates
$stmt = $db->prepare("
    SELECT id, title, issuing_organization, issue_date, expiry_date, credential_id, 
           credential_url, description, file_path, created_at
    FROM certificates 
    WHERE candidate_id = ?
    ORDER BY issue_date DESC, created_at DESC
");
$stmt->execute([$candidate_id]);
$candidate_certificates = $stmt->fetchAll();

// Get candidate's other applications
$stmt = $db->prepare("
    SELECT ja.status, ja.applied_at, j.title as job_title, c.name as company_name
    FROM job_applications ja
    JOIN jobs j ON ja.job_id = j.id
    JOIN companies c ON j.company_id = c.id
    WHERE ja.candidate_id = ? AND ja.id != ?
    ORDER BY ja.applied_at DESC
    LIMIT 5
");
$stmt->execute([$candidate_id, $application_id]);
$other_applications = $stmt->fetchAll();

// Handle status updates
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];
    $notes = isset($_POST['notes']) ? sanitize($_POST['notes']) : '';
    
    // Store old status for notification
    $old_status = $application['status'];
    
    // Check if shortlisting is allowed (only after deadline)
    if ($action === 'shortlist') {
        $job_deadline = $application['deadline'];
        $current_date = date('Y-m-d');
        
        if ($job_deadline && $current_date <= $job_deadline) {
            $error = 'Shortlisting is only allowed after the application deadline (' . formatDate($job_deadline) . ').';
        } else {
            $new_status = 'shortlisted';
        }
    } else {
        // For other actions, proceed normally
        switch ($action) {
            case 'reject':
                $new_status = 'rejected';
                break;
            case 'interview':
                $new_status = 'interviewed';
                break;
            case 'hire':
                $new_status = 'hired';
                break;
        }
    }
    
    if (isset($new_status) && $new_status && !$error) {
        try {
            $stmt = $db->prepare("UPDATE job_applications SET status = ? WHERE id = ?");
            $stmt->execute([$new_status, $application_id]);
            
            // Send notification to candidate
            require_once '../includes/notification_system.php';
            $notificationSystem = new NotificationSystem($db);
            $notification_sent = $notificationSystem->notifyApplicationStatusChange($application_id, $old_status, $new_status);
            
            if ($notification_sent) {
                $success = 'Application status updated successfully! Candidate has been notified.';
            } else {
                $success = 'Application status updated successfully! (Note: Notification system may have encountered issues - check logs)';
            }
            
            // Refresh application data
            $application['status'] = $new_status;
            
        } catch (PDOException $e) {
            $error = 'Error updating application status.';
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Application Details - <?php echo SITE_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
    <style>
        .profile-header {
            background: linear-gradient(135deg, #0d6efd, #0056b3);
            color: white;
            padding: 2rem 0;
            margin-bottom: 2rem;
        }
        .profile-picture-large {
            width: 120px;
            height: 120px;
            object-fit: cover;
            border-radius: 50%;
            border: 4px solid white;
        }
        .section-card {
            border: none;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 1.5rem;
        }
        .section-header {
            background-color: #f8f9fa;
            border-bottom: 1px solid #dee2e6;
            padding: 1rem 1.5rem;
            margin: 0;
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container-fluid">
            <a class="navbar-brand" href="../index.php">
                <i class="fas fa-briefcase me-2"></i><?php echo SITE_NAME; ?>
            </a>
            
            <div class="navbar-nav ms-auto">
                <span class="navbar-text me-3">
                    Welcome, <?php echo htmlspecialchars($_SESSION['company_name']); ?>
                </span>
                <a class="nav-link" href="../auth/logout.php">
                    <i class="fas fa-sign-out-alt me-1"></i>Logout
                </a>
            </div>
        </div>
    </nav>

    <!-- Breadcrumb -->
    <div class="container-fluid">
        <div class="row">
            <div class="col-12">
                <nav aria-label="breadcrumb" class="pt-3">
                    <ol class="breadcrumb">
                        <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
                        <li class="breadcrumb-item"><a href="applications.php">Applications</a></li>
                        <li class="breadcrumb-item active">Application Details</li>
                    </ol>
                </nav>
            </div>
        </div>
    </div>

    <!-- Profile Header -->
    <div class="profile-header">
        <div class="container">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <div class="d-flex align-items-center">
                        <div class="me-4">
                            <?php if ($application['profile_picture']): ?>
                                <img src="<?php echo UPLOAD_URL . 'profiles/' . $application['profile_picture']; ?>" 
                                     alt="Profile Picture" class="profile-picture-large">
                            <?php else: ?>
                                <div class="profile-picture-large bg-light d-flex align-items-center justify-content-center">
                                    <i class="fas fa-user fa-3x text-muted"></i>
                                </div>
                            <?php endif; ?>
                        </div>
                        <div>
                            <h1 class="h2 mb-2"><?php echo htmlspecialchars($application['first_name'] . ' ' . $application['last_name']); ?></h1>
                            <p class="mb-1 opacity-75">Applied for: <strong><?php echo htmlspecialchars($application['job_title']); ?></strong></p>
                            <p class="mb-1 opacity-75">
                                <i class="fas fa-envelope me-2"></i><?php echo htmlspecialchars($application['email']); ?>
                            </p>
                            <?php if ($application['phone']): ?>
                                <p class="mb-0 opacity-75">
                                    <i class="fas fa-phone me-2"></i><?php echo htmlspecialchars($application['phone']); ?>
                                </p>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                <div class="col-md-4 text-md-end">
                    <div class="mb-3">
                        <span class="badge status-<?php echo $application['status']; ?> badge-lg fs-6 px-3 py-2">
                            <?php echo ucfirst($application['status']); ?>
                        </span>
                    </div>
                    <p class="mb-1 opacity-75">Applied: <?php echo formatDate($application['applied_at']); ?></p>
                    <?php if ($application['deadline']): ?>
                        <p class="mb-1 opacity-75">Application Deadline: <?php echo formatDate($application['deadline']); ?></p>
                    <?php endif; ?>
                    <p class="mb-0 opacity-75">Member since: <?php echo formatDate($application['candidate_since']); ?></p>
                </div>
            </div>
        </div>
    </div>

    <div class="container">
        <?php if ($error): ?>
            <div class="alert alert-danger">
                <i class="fas fa-exclamation-circle me-2"></i><?php echo $error; ?>
            </div>
        <?php endif; ?>

        <?php if ($success): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
            </div>
        <?php endif; ?>

        <div class="row">
            <!-- Main Content -->
            <div class="col-lg-8">
                <!-- Cover Letter -->
                <div class="card section-card">
                    <div class="section-header">
                        <h5 class="mb-0"><i class="fas fa-envelope me-2"></i>Cover Letter</h5>
                    </div>
                    <div class="card-body">
                        <div class="content">
                            <?php echo nl2br(htmlspecialchars($application['cover_letter'])); ?>
                        </div>
                    </div>
                </div>

                <!-- Survey Responses -->
                <?php if (!empty($survey_responses)): ?>
                    <div class="card section-card">
                        <div class="section-header">
                            <h5 class="mb-0"><i class="fas fa-question-circle me-2"></i>Additional Questions & Responses</h5>
                        </div>
                        <div class="card-body">
                            <?php foreach ($survey_responses as $response): ?>
                                <div class="mb-4">
                                    <h6 class="text-primary mb-2"><?php echo htmlspecialchars($response['question']); ?></h6>
                                    <div class="bg-light p-3 rounded">
                                        <?php echo nl2br(htmlspecialchars($response['response'])); ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endif; ?>

                <!-- Resume Summary -->
                <?php if ($application['resume_title']): ?>
                    <div class="card section-card">
                        <div class="section-header">
                            <h5 class="mb-0"><i class="fas fa-file-alt me-2"></i>Resume: <?php echo htmlspecialchars($application['resume_title']); ?></h5>
                        </div>
                        <div class="card-body">
                            <?php if ($application['summary']): ?>
                                <div class="mb-3">
                                    <h6>Professional Summary</h6>
                                    <p><?php echo nl2br(htmlspecialchars($application['summary'])); ?></p>
                                </div>
                            <?php endif; ?>
                            
                            <?php if ($application['skills']): ?>
                                <div>
                                    <h6>Skills</h6>
                                    <p><?php echo nl2br(htmlspecialchars($application['skills'])); ?></p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endif; ?>

                <!-- Work Experience -->
                <?php if (!empty($experience)): ?>
                    <div class="card section-card">
                        <div class="section-header">
                            <h5 class="mb-0"><i class="fas fa-briefcase me-2"></i>Work Experience</h5>
                        </div>
                        <div class="card-body">
                            <?php foreach ($experience as $exp): ?>
                                <div class="experience-item border-start border-3 border-primary ps-3 mb-4">
                                    <h6 class="mb-1"><?php echo htmlspecialchars($exp['position']); ?></h6>
                                    <p class="text-primary mb-2"><?php echo htmlspecialchars($exp['company_name']); ?></p>
                                    <p class="text-muted mb-2">
                                        <?php if ($exp['start_date']): ?>
                                            <?php echo formatDate($exp['start_date']); ?>
                                        <?php endif; ?>
                                        - 
                                        <?php if ($exp['is_current']): ?>
                                            Present
                                        <?php elseif ($exp['end_date']): ?>
                                            <?php echo formatDate($exp['end_date']); ?>
                                        <?php endif; ?>
                                    </p>
                                    <?php if ($exp['description']): ?>
                                        <p><?php echo nl2br(htmlspecialchars($exp['description'])); ?></p>
                                    <?php endif; ?>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endif; ?>

                <!-- Education -->
                <?php if (!empty($education)): ?>
                    <div class="card section-card">
                        <div class="section-header">
                            <h5 class="mb-0"><i class="fas fa-graduation-cap me-2"></i>Education</h5>
                        </div>
                        <div class="card-body">
                            <?php foreach ($education as $edu): ?>
                                <div class="education-item border-start border-3 border-info ps-3 mb-4">
                                    <h6 class="mb-1"><?php echo htmlspecialchars($edu['degree']); ?></h6>
                                    <?php if ($edu['field_of_study']): ?>
                                        <p class="text-muted mb-1"><?php echo htmlspecialchars($edu['field_of_study']); ?></p>
                                    <?php endif; ?>
                                    <p class="text-info mb-2"><?php echo htmlspecialchars($edu['institution']); ?></p>
                                    <p class="text-muted mb-2">
                                        <?php if ($edu['start_date']): ?>
                                            <?php echo formatDate($edu['start_date']); ?>
                                        <?php endif; ?>
                                        <?php if ($edu['end_date']): ?>
                                            - <?php echo formatDate($edu['end_date']); ?>
                                        <?php endif; ?>
                                        <?php if ($edu['grade']): ?>
                                            • Grade: <?php echo htmlspecialchars($edu['grade']); ?>
                                        <?php endif; ?>
                                    </p>
                                    <?php if ($edu['description']): ?>
                                        <p><?php echo nl2br(htmlspecialchars($edu['description'])); ?></p>
                                    <?php endif; ?>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endif; ?>

                <!-- Documents & Attachments -->
                <?php if (!empty($candidate_documents)): ?>
                    <div class="card section-card">
                        <div class="section-header">
                            <h5 class="mb-0"><i class="fas fa-paperclip me-2"></i>Documents & Attachments</h5>
                        </div>
                        <div class="card-body">
                            <?php 
                            // Group documents by type
                            $grouped_docs = [];
                            foreach ($candidate_documents as $doc) {
                                $grouped_docs[$doc['document_type']][] = $doc;
                            }
                            ?>
                            
                            <?php foreach ($grouped_docs as $doc_type => $documents): ?>
                                <div class="mb-4">
                                    <h6 class="text-primary mb-3">
                                        <i class="fas fa-<?php echo getDocumentIcon($doc_type); ?> me-2"></i>
                                        <?php echo ucfirst(str_replace('_', ' ', $doc_type)); ?>s
                                    </h6>
                                    
                                    <?php foreach ($documents as $doc): ?>
                                        <div class="d-flex align-items-center justify-content-between p-3 mb-2 border rounded bg-light">
                                            <div class="d-flex align-items-center">
                                                <div class="me-3">
                                                    <i class="fas fa-file-<?php echo getFileTypeIcon($doc['file_type']); ?> fa-2x text-<?php echo getFileTypeColor($doc['file_type']); ?>"></i>
                                                </div>
                                                <div>
                                                    <h6 class="mb-1"><?php echo htmlspecialchars($doc['title']); ?></h6>
                                                    <small class="text-muted">
                                                        <?php echo formatFileSize($doc['file_size']); ?> • 
                                                        <?php echo formatDate($doc['created_at']); ?>
                                                    </small>
                                                    <?php if ($doc['description']): ?>
                                                        <p class="mb-0 mt-1 text-muted small"><?php echo htmlspecialchars($doc['description']); ?></p>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                            <div class="btn-group btn-group-sm">
                                                <a href="download-document.php?id=<?php echo $doc['id']; ?>&type=document" 
                                                   class="btn btn-outline-primary" title="Download">
                                                    <i class="fas fa-download"></i>
                                                </a>
                                                <a href="view-document.php?id=<?php echo $doc['id']; ?>&type=document" 
                                                   class="btn btn-outline-secondary" title="View" target="_blank">
                                                    <i class="fas fa-eye"></i>
                                                </a>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endif; ?>

                <!-- Certificates -->
                <?php if (!empty($candidate_certificates)): ?>
                    <div class="card section-card">
                        <div class="section-header">
                            <h5 class="mb-0"><i class="fas fa-certificate me-2"></i>Certificates & Credentials</h5>
                        </div>
                        <div class="card-body">
                            <?php foreach ($candidate_certificates as $cert): ?>
                                <div class="certificate-item border-start border-3 border-warning ps-3 mb-4">
                                    <div class="d-flex justify-content-between align-items-start">
                                        <div class="flex-grow-1">
                                            <h6 class="mb-1"><?php echo htmlspecialchars($cert['title']); ?></h6>
                                            <?php if ($cert['issuing_organization']): ?>
                                                <p class="text-warning mb-2"><?php echo htmlspecialchars($cert['issuing_organization']); ?></p>
                                            <?php endif; ?>
                                            
                                            <div class="row text-muted mb-2">
                                                <?php if ($cert['issue_date']): ?>
                                                    <div class="col-sm-6">
                                                        <small><i class="fas fa-calendar-alt me-1"></i>Issued: <?php echo formatDate($cert['issue_date']); ?></small>
                                                    </div>
                                                <?php endif; ?>
                                                <?php if ($cert['expiry_date']): ?>
                                                    <div class="col-sm-6">
                                                        <small><i class="fas fa-clock me-1"></i>Expires: <?php echo formatDate($cert['expiry_date']); ?></small>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                            
                                            <?php if ($cert['credential_id']): ?>
                                                <p class="mb-2">
                                                    <small class="text-muted">Credential ID: </small>
                                                    <code><?php echo htmlspecialchars($cert['credential_id']); ?></code>
                                                </p>
                                            <?php endif; ?>
                                            
                                            <?php if ($cert['description']): ?>
                                                <p class="mb-2"><?php echo nl2br(htmlspecialchars($cert['description'])); ?></p>
                                            <?php endif; ?>
                                            
                                            <div class="btn-group btn-group-sm">
                                                <?php if ($cert['file_path']): ?>
                                                    <a href="download-document.php?id=<?php echo $cert['id']; ?>&type=certificate" 
                                                       class="btn btn-outline-primary btn-sm" title="Download Certificate">
                                                        <i class="fas fa-download me-1"></i>Download
                                                    </a>
                                                    <a href="view-document.php?id=<?php echo $cert['id']; ?>&type=certificate" 
                                                       class="btn btn-outline-secondary btn-sm" title="View Certificate" target="_blank">
                                                        <i class="fas fa-eye me-1"></i>View
                                                    </a>
                                                <?php endif; ?>
                                                <?php if ($cert['credential_url']): ?>
                                                    <a href="<?php echo htmlspecialchars($cert['credential_url']); ?>" 
                                                       class="btn btn-outline-info btn-sm" title="Verify Online" target="_blank">
                                                        <i class="fas fa-external-link-alt me-1"></i>Verify
                                                    </a>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Sidebar -->
            <div class="col-lg-4">
                <!-- Actions -->
                <div class="card section-card">
                    <div class="section-header">
                        <h6 class="mb-0"><i class="fas fa-cogs me-2"></i>Actions</h6>
                    </div>
                    <div class="card-body">
                        <?php if ($application['deadline'] && !$shortlisting_allowed): ?>
                            <div class="alert alert-info">
                                <i class="fas fa-clock me-2"></i>
                                <strong>Deadline Status:</strong><br>
                                <?php echo $deadline_message; ?>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($application['status'] === 'applied'): ?>
                            <div class="d-grid gap-2">
                                <form method="POST" class="d-inline">
                                    <input type="hidden" name="action" value="shortlist">
                                    <button type="submit" class="btn btn-warning w-100" <?php echo !$shortlisting_allowed ? 'disabled' : ''; ?>>
                                        <i class="fas fa-star me-2"></i>Shortlist Candidate
                                        <?php if (!$shortlisting_allowed): ?>
                                            <small class="d-block">(Available after deadline)</small>
                                        <?php endif; ?>
                                    </button>
                                </form>
                                
                                <a href="schedule-interview.php?application_id=<?php echo $application['id']; ?>" class="btn btn-info">
                                    <i class="fas fa-calendar-plus me-2"></i>Schedule Interview
                                </a>
                                
                                <form method="POST" class="d-inline" onsubmit="return confirm('Are you sure you want to reject this application?')">
                                    <input type="hidden" name="action" value="reject">
                                    <button type="submit" class="btn btn-outline-danger w-100">
                                        <i class="fas fa-times me-2"></i>Reject Application
                                    </button>
                                </form>
                            </div>
                        <?php elseif ($application['status'] === 'shortlisted'): ?>
                            <div class="d-grid gap-2">
                                <a href="schedule-interview.php?application_id=<?php echo $application['id']; ?>" class="btn btn-info">
                                    <i class="fas fa-calendar-plus me-2"></i>Schedule Interview
                                </a>
                                
                                <form method="POST" class="d-inline" onsubmit="return confirm('Are you sure you want to reject this application?')">
                                    <input type="hidden" name="action" value="reject">
                                    <button type="submit" class="btn btn-outline-danger w-100">
                                        <i class="fas fa-times me-2"></i>Reject Application
                                    </button>
                                </form>
                            </div>
                        <?php elseif ($application['status'] === 'interviewed'): ?>
                            <div class="d-grid gap-2">
                                <form method="POST" class="d-inline">
                                    <input type="hidden" name="action" value="hire">
                                    <button type="submit" class="btn btn-success w-100">
                                        <i class="fas fa-check me-2"></i>Hire Candidate
                                    </button>
                                </form>
                                
                                <form method="POST" class="d-inline" onsubmit="return confirm('Are you sure you want to reject this application?')">
                                    <input type="hidden" name="action" value="reject">
                                    <button type="submit" class="btn btn-outline-danger w-100">
                                        <i class="fas fa-times me-2"></i>Reject Application
                                    </button>
                                </form>
                            </div>
                        <?php elseif ($application['status'] === 'hired'): ?>
                            <div class="alert alert-success text-center">
                                <i class="fas fa-check-circle fa-2x mb-2"></i>
                                <h6>Candidate Hired</h6>
                                <p class="mb-0">This candidate has been successfully hired.</p>
                            </div>
                        <?php elseif ($application['status'] === 'rejected'): ?>
                            <div class="alert alert-danger text-center">
                                <i class="fas fa-times-circle fa-2x mb-2"></i>
                                <h6>Application Rejected</h6>
                                <p class="mb-0">This application has been rejected.</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Candidate Information -->
                <div class="card section-card">
                    <div class="section-header">
                        <h6 class="mb-0"><i class="fas fa-user me-2"></i>Candidate Information</h6>
                    </div>
                    <div class="card-body">
                        <?php if ($application['address']): ?>
                            <div class="mb-3">
                                <small class="text-muted">Address</small>
                                <div><?php echo htmlspecialchars($application['address']); ?></div>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($application['date_of_birth']): ?>
                            <div class="mb-3">
                                <small class="text-muted">Date of Birth</small>
                                <div><?php echo formatDate($application['date_of_birth']); ?></div>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($application['gender']): ?>
                            <div class="mb-3">
                                <small class="text-muted">Gender</small>
                                <div><?php echo ucfirst($application['gender']); ?></div>
                            </div>
                        <?php endif; ?>
                        
                        <div class="mb-0">
                            <small class="text-muted">Member Since</small>
                            <div><?php echo formatDate($application['candidate_since']); ?></div>
                        </div>
                    </div>
                </div>

                <!-- Other Applications -->
                <?php if (!empty($other_applications)): ?>
                    <div class="card section-card">
                        <div class="section-header">
                            <h6 class="mb-0"><i class="fas fa-list me-2"></i>Other Applications</h6>
                        </div>
                        <div class="card-body">
                            <?php foreach ($other_applications as $other_app): ?>
                                <div class="mb-3 pb-3 <?php echo $other_app !== end($other_applications) ? 'border-bottom' : ''; ?>">
                                    <div class="d-flex justify-content-between align-items-start">
                                        <div>
                                            <h6 class="mb-1"><?php echo htmlspecialchars($other_app['job_title']); ?></h6>
                                            <small class="text-muted"><?php echo htmlspecialchars($other_app['company_name']); ?></small>
                                        </div>
                                        <span class="badge status-<?php echo $other_app['status']; ?>">
                                            <?php echo ucfirst($other_app['status']); ?>
                                        </span>
                                    </div>
                                    <small class="text-muted"><?php echo formatDate($other_app['applied_at']); ?></small>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endif; ?>

                <!-- Back Button -->
                <div class="d-grid">
                    <a href="applications.php" class="btn btn-outline-secondary">
                        <i class="fas fa-arrow-left me-2"></i>Back to Applications
                    </a>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

<?php
// Helper functions for document display
function getDocumentIcon($doc_type) {
    switch($doc_type) {
        case 'cv': return 'user-tie';
        case 'cover_letter': return 'envelope';
        case 'certificate': return 'certificate';
        case 'transcript': return 'graduation-cap';
        case 'other': return 'file-alt';
        default: return 'file';
    }
}

function getFileTypeIcon($file_type) {
    if (strpos($file_type, 'pdf') !== false) return 'pdf';
    if (strpos($file_type, 'word') !== false || strpos($file_type, 'document') !== false) return 'word';
    if (strpos($file_type, 'image') !== false) return 'image';
    if (strpos($file_type, 'text') !== false) return 'alt';
    return 'alt';
}

function getFileTypeColor($file_type) {
    if (strpos($file_type, 'pdf') !== false) return 'danger';
    if (strpos($file_type, 'word') !== false || strpos($file_type, 'document') !== false) return 'primary';
    if (strpos($file_type, 'image') !== false) return 'success';
    if (strpos($file_type, 'text') !== false) return 'info';
    return 'secondary';
}

function formatFileSize($bytes) {
    if (!$bytes) return 'Unknown size';
    $units = array('B', 'KB', 'MB', 'GB');
    for ($i = 0; $bytes >= 1024 && $i < count($units) - 1; $i++) {
        $bytes /= 1024;
    }
    return round($bytes, 2) . ' ' . $units[$i];
}
?>