<?php
echo "<h1>PDF Generation Diagnostic Test</h1>";

// Test 1: Check TCPDF availability
echo "<h2>1. TCPDF Availability Check</h2>";
if (class_exists('TCPDF')) {
    echo "<p style='color: green;'>✓ TCPDF class is available</p>";
} else {
    echo "<p style='color: orange;'>⚠ TCPDF class not found - checking for file</p>";
    
    $tcpdf_paths = [
        dirname(__FILE__) . '/includes/tcpdf/tcpdf.php',
        dirname(__FILE__) . '/tcpdf/tcpdf.php',
        'includes/tcpdf/tcpdf.php'
    ];
    
    $tcpdf_found = false;
    foreach ($tcpdf_paths as $path) {
        if (file_exists($path)) {
            echo "<p style='color: green;'>✓ TCPDF found at: {$path}</p>";
            $tcpdf_found = true;
            break;
        }
    }
    
    if (!$tcpdf_found) {
        echo "<p style='color: red;'>✗ TCPDF not found in common locations</p>";
    }
}

// Test 2: Check wkhtmltopdf availability
echo "<h2>2. wkhtmltopdf Availability Check</h2>";
if (function_exists('exec')) {
    exec('wkhtmltopdf --version 2>&1', $output, $return_var);
    if ($return_var === 0) {
        echo "<p style='color: green;'>✓ wkhtmltopdf is available</p>";
        echo "<pre>" . implode("\n", $output) . "</pre>";
    } else {
        echo "<p style='color: red;'>✗ wkhtmltopdf not available</p>";
    }
} else {
    echo "<p style='color: red;'>✗ exec() function not available</p>";
}

// Test 3: Test basic HTML to PDF conversion
echo "<h2>3. Basic PDF Generation Test</h2>";

$test_html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Test CV</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        h1 { color: #2c3e50; }
    </style>
</head>
<body>
    <h1>Test Curriculum Vitae</h1>
    <p>This is a test PDF document to verify PDF generation is working.</p>
    <p>Generated on: ' . date('F j, Y \a\t g:i A') . '</p>
</body>
</html>';

function generateSimplePDF($html) {
    // Try wkhtmltopdf first
    if (function_exists('exec')) {
        exec('wkhtmltopdf --version 2>&1', $output, $return_var);
        if ($return_var === 0) {
            $tempHtml = tempnam(sys_get_temp_dir(), 'test_cv_') . '.html';
            $tempPdf = tempnam(sys_get_temp_dir(), 'test_cv_') . '.pdf';
            
            file_put_contents($tempHtml, $html);
            
            $command = "wkhtmltopdf --page-size A4 --quiet \"$tempHtml\" \"$tempPdf\" 2>/dev/null";
            exec($command, $output, $return_var);
            
            if ($return_var === 0 && file_exists($tempPdf)) {
                $pdfContent = file_get_contents($tempPdf);
                unlink($tempHtml);
                unlink($tempPdf);
                return $pdfContent;
            }
            
            // Clean up on failure
            if (file_exists($tempHtml)) unlink($tempHtml);
            if (file_exists($tempPdf)) unlink($tempPdf);
        }
    }
    
    return false;
}

$pdf_content = generateSimplePDF($test_html);

if ($pdf_content !== false) {
    echo "<p style='color: green;'>✓ PDF generation successful!</p>";
    echo "<p>PDF size: " . number_format(strlen($pdf_content)) . " bytes</p>";
    
    // Save test PDF
    $test_pdf_file = 'test_cv_generation.pdf';
    file_put_contents($test_pdf_file, $pdf_content);
    echo "<p><a href='{$test_pdf_file}' style='background: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 4px;' download>📄 Download Test PDF</a></p>";
    
    // Check if PDF is valid
    $pdf_header = substr($pdf_content, 0, 4);
    if ($pdf_header === '%PDF') {
        echo "<p style='color: green;'>✓ PDF has valid header</p>";
    } else {
        echo "<p style='color: red;'>✗ PDF header is invalid: " . htmlspecialchars($pdf_header) . "</p>";
    }
    
} else {
    echo "<p style='color: red;'>✗ PDF generation failed</p>";
    echo "<p>Falling back to HTML generation...</p>";
    
    // Save as HTML instead
    $test_html_file = 'test_cv_generation.html';
    file_put_contents($test_html_file, $test_html);
    echo "<p><a href='{$test_html_file}' style='background: #28a745; color: white; padding: 10px 20px; text-decoration: none; border-radius: 4px;' target='_blank'>🌐 View HTML Version</a></p>";
    echo "<p><em>You can use your browser's Print function to save as PDF</em></p>";
}

// Test 4: Check temp directory permissions
echo "<h2>4. System Requirements Check</h2>";
$temp_dir = sys_get_temp_dir();
echo "<p><strong>Temp directory:</strong> {$temp_dir}</p>";

if (is_writable($temp_dir)) {
    echo "<p style='color: green;'>✓ Temp directory is writable</p>";
} else {
    echo "<p style='color: red;'>✗ Temp directory is not writable</p>";
}

// Test creating a temp file
$test_temp = tempnam($temp_dir, 'test_');
if ($test_temp) {
    echo "<p style='color: green;'>✓ Can create temp files</p>";
    unlink($test_temp);
} else {
    echo "<p style='color: red;'>✗ Cannot create temp files</p>";
}

echo "<h2>5. Recommendations</h2>";
echo "<div style='background: #e7f3ff; padding: 15px; border-left: 4px solid #007bff;'>";

if ($pdf_content !== false) {
    echo "<h4 style='color: green;'>✅ PDF Generation Working!</h4>";
    echo "<p>Your system can generate PDFs successfully. The CV builder will create proper PDF files.</p>";
} else {
    echo "<h4 style='color: orange;'>⚠ PDF Generation Not Available</h4>";
    echo "<p>PDF generation failed. Recommendations:</p>";
    echo "<ul>";
    echo "<li><strong>Option 1:</strong> Install wkhtmltopdf on your system</li>";
    echo "<li><strong>Option 2:</strong> Install TCPDF library</li>";
    echo "<li><strong>Option 3:</strong> Use HTML fallback (browser can save as PDF)</li>";
    echo "</ul>";
    
    echo "<h4>Installation Instructions:</h4>";
    echo "<p><strong>For Windows (XAMPP):</strong></p>";
    echo "<ol>";
    echo "<li>Download wkhtmltopdf from <a href='https://wkhtmltopdf.org/downloads.html' target='_blank'>https://wkhtmltopdf.org/downloads.html</a></li>";
    echo "<li>Install it and add to PATH</li>";
    echo "<li>Or place wkhtmltopdf.exe in your project directory</li>";
    echo "</ol>";
    
    echo "<p><strong>For TCPDF:</strong></p>";
    echo "<ol>";
    echo "<li>Download TCPDF from <a href='https://tcpdf.org/' target='_blank'>https://tcpdf.org/</a></li>";
    echo "<li>Extract to includes/tcpdf/ directory</li>";
    echo "<li>Ensure tcpdf.php is at includes/tcpdf/tcpdf.php</li>";
    echo "</ol>";
}

echo "</div>";
?>