<?php
require_once 'config/config.php';
require_once 'includes/pdf_cv_generator.php';

$database = new Database();
$db = $database->getConnection();

// Test with Sisanda who has good data
$candidate_id = 13;

echo "Testing modern CV layout for candidate ID: $candidate_id\n";

// Get application data
$stmt = $db->prepare("
    SELECT ja.id, ja.candidate_id, ja.resume_id, ja.cover_letter, ja.applied_at, ja.status, 
           COALESCE(ja.total_score, 0) as total_score,
           j.title as job_title, j.description as job_description,
           c.first_name, c.last_name, c.email, c.profile_picture,
           r.title as resume_title, r.file_path as resume_path
    FROM job_applications ja
    JOIN jobs j ON ja.job_id = j.id
    JOIN candidates c ON ja.candidate_id = c.id
    LEFT JOIN resumes r ON ja.resume_id = r.id
    WHERE ja.candidate_id = ? AND j.title LIKE '%Business Development%'
    LIMIT 1
");
$stmt->execute([$candidate_id]);
$app = $stmt->fetch();

if (!$app) {
    echo "No application found\n";
    exit;
}

// Get profile data
$stmt = $db->prepare("SELECT * FROM candidates WHERE id = ?");
$stmt->execute([$candidate_id]);
$profile_data = $stmt->fetch();

// Remove test overrides - use real database data
// The candidate has sa_id_number: 9206295469083 and is_foreign_national: no

echo "Testing with real database data:\n";
echo "sa_id_number: " . ($profile_data['sa_id_number'] ?? 'NULL') . "\n";
echo "is_foreign_national: " . ($profile_data['is_foreign_national'] ?? 'NULL') . "\n";
echo "nationality: " . ($profile_data['nationality'] ?? 'NULL') . "\n";

// Get CV data
$stmt = $db->prepare("SELECT * FROM education WHERE candidate_id = ? ORDER BY start_date DESC");
$stmt->execute([$candidate_id]);
$education = $stmt->fetchAll();

$stmt = $db->prepare("SELECT * FROM work_experience WHERE candidate_id = ? ORDER BY start_date DESC");
$stmt->execute([$candidate_id]);
$experience = $stmt->fetchAll();

try {
    $stmt = $db->prepare("SELECT * FROM work_references WHERE candidate_id = ? ORDER BY created_at DESC");
    $stmt->execute([$candidate_id]);
    $references = $stmt->fetchAll();
} catch (PDOException $e) {
    $references = [];
}

$cv_data = [
    'education' => $education,
    'experience' => $experience,
    'references' => $references,
    'skills' => [
        ['skill_name' => 'Business Development'],
        ['skill_name' => 'Sales Management'],
        ['skill_name' => 'Customer Relations'],
        ['skill_name' => 'Strategic Planning'],
        ['skill_name' => 'Market Analysis'],
        ['skill_name' => 'Team Leadership']
    ]
];

// Get qualifications
$stmt = $db->prepare("SELECT * FROM certificates WHERE candidate_id = ? ORDER BY issue_date DESC");
$stmt->execute([$candidate_id]);
$qualifications = ['certificates' => $stmt->fetchAll()];

echo "Candidate: {$app['first_name']} {$app['last_name']}\n";
echo "Features to test:\n";
echo "- Modern header with gradient colors\n";
echo "- Application summary with position on own row\n";
echo "- Profile introduction section\n";
echo "- ID number: " . ($profile_data['sa_id_number'] ?? $profile_data['id_number'] ?? 'None') . "\n";

// Test the nationality determination
$generator = new PDFCVGenerator();
$reflection = new ReflectionClass($generator);
$method = $reflection->getMethod('determineNationality');
$method->setAccessible(true);
$determined_nationality = $method->invoke($generator, $profile_data);

echo "- Determined nationality: $determined_nationality\n";
echo "- Modern skills grid layout\n";

try {
    $pdf_generator = new PDFCVGenerator();
    $pdf_content = $pdf_generator->generateCV($app, $cv_data, $profile_data, $qualifications);
    
    $filename = "modern_cv_" . $app['first_name'] . "_" . $app['last_name'] . ".pdf";
    file_put_contents($filename, $pdf_content);
    
    echo "\nModern CV generated successfully: $filename\n";
    echo "File size: " . strlen($pdf_content) . " bytes\n";
    echo "\nNew features included:\n";
    echo "✓ Modern gradient header\n";
    echo "✓ Application summary with position on separate row\n";
    echo "✓ Profile introduction with dynamic text\n";
    echo "✓ ID number in application summary\n";
    echo "✓ Modern skills grid with colored boxes\n";
    echo "✓ Enhanced typography and spacing\n";
    
} catch (Exception $e) {
    echo "Error generating PDF: " . $e->getMessage() . "\n";
}
?>