<?php
require_once 'fpdf.php';

class PDFCVGenerator {
    private $pdf;
    private $primaryColor = [26, 35, 126]; // Deep blue
    private $accentColor = [63, 81, 181]; // Modern blue
    private $lightAccent = [121, 134, 203]; // Light blue
    private $textColor = [33, 37, 41]; // Dark gray
    private $lightGray = [248, 249, 250]; // Very light gray
    private $mediumGray = [108, 117, 125]; // Medium gray
    
    public function __construct() {
        $this->pdf = new FPDF('P', 'mm', 'A4');
        $this->pdf->SetAutoPageBreak(true, 20);
        $this->pdf->AddPage();
    }
    
    public function generateCV($app, $cv_data, $profile_data, $qualifications = []) {
        // Add modern header with profile picture and name
        $this->addModernHeader($app, $profile_data);
        
        // Add application summary at top
        $this->addApplicationSummary($app, $profile_data);
        
        // Add profile introduction
        $this->addProfileIntroduction($app, $profile_data);
        
        // Add contact information in modern layout
        $this->addContactInfo($app, $profile_data);
        
        // Add sections with modern styling
        $this->addSkillsSection($cv_data);
        $this->addEducationSection($cv_data);
        $this->addCertificationsSection($qualifications);
        $this->addExperienceSection($cv_data);
        $this->addReferencesSection($cv_data);
        
        return $this->pdf->Output('S'); // Return PDF as string
    }
    
    private function addHeader($app, $profile_data) {
        // Header background
        $this->pdf->SetFillColor($this->primaryColor[0], $this->primaryColor[1], $this->primaryColor[2]);
        $this->pdf->Rect(0, 0, 210, 50, 'F');
        
        // Profile picture placeholder (if available)
        $profile_x = 160;
        $profile_y = 10;
        $profile_size = 30;
        
        if (!empty($app['profile_picture'])) {
            $profile_path = UPLOAD_PATH . 'profiles/' . $app['profile_picture'];
            if (file_exists($profile_path)) {
                try {
                    // Add profile picture
                    $this->pdf->Image($profile_path, $profile_x, $profile_y, $profile_size, $profile_size);
                } catch (Exception $e) {
                    // If image fails, draw placeholder
                    $this->drawProfilePlaceholder($profile_x, $profile_y, $profile_size);
                }
            } else {
                $this->drawProfilePlaceholder($profile_x, $profile_y, $profile_size);
            }
        } else {
            $this->drawProfilePlaceholder($profile_x, $profile_y, $profile_size);
        }
        
        // Name
        $this->pdf->SetFont('Arial', 'B', 24);
        $this->pdf->SetTextColor(255, 255, 255);
        $this->pdf->SetXY(20, 15);
        $candidate_name = trim($app['first_name'] . ' ' . $app['last_name']);
        $this->pdf->Cell(120, 10, $candidate_name, 0, 1, 'L');
        
        // Position applied for
        $this->pdf->SetFont('Arial', '', 12);
        $this->pdf->SetXY(20, 30);
        $this->pdf->Cell(120, 8, 'Position Applied For: ' . $app['job_title'], 0, 1, 'L');
        
        $this->pdf->Ln(25);
    }
    
    private function drawProfilePlaceholder($x, $y, $size) {
        $this->pdf->SetFillColor(255, 255, 255);
        $this->pdf->Rect($x, $y, $size, $size, 'F');
        $this->pdf->SetDrawColor(200, 200, 200);
        $this->pdf->Rect($x, $y, $size, $size, 'D');
    }
    
    private function addModernHeader($app, $profile_data) {
        // Modern gradient-style header
        $this->pdf->SetFillColor($this->primaryColor[0], $this->primaryColor[1], $this->primaryColor[2]);
        $this->pdf->Rect(0, 0, 210, 45, 'F');
        
        // Accent stripe
        $this->pdf->SetFillColor($this->accentColor[0], $this->accentColor[1], $this->accentColor[2]);
        $this->pdf->Rect(0, 40, 210, 5, 'F');
        
        // Profile picture with modern styling
        $profile_x = 155;
        $profile_y = 8;
        $profile_size = 28;
        
        if (!empty($app['profile_picture'])) {
            $profile_path = UPLOAD_PATH . 'profiles/' . $app['profile_picture'];
            if (file_exists($profile_path)) {
                try {
                    // White background circle for profile (using rect as fallback)
                    $this->pdf->SetFillColor(255, 255, 255);
                    $this->pdf->Rect($profile_x - 2, $profile_y - 2, $profile_size + 4, $profile_size + 4, 'F');
                    
                    $this->pdf->Image($profile_path, $profile_x, $profile_y, $profile_size, $profile_size);
                } catch (Exception $e) {
                    $this->drawModernProfilePlaceholder($profile_x, $profile_y, $profile_size);
                }
            } else {
                $this->drawModernProfilePlaceholder($profile_x, $profile_y, $profile_size);
            }
        } else {
            $this->drawModernProfilePlaceholder($profile_x, $profile_y, $profile_size);
        }
        
        // Name with modern typography
        $this->pdf->SetFont('Arial', 'B', 22);
        $this->pdf->SetTextColor(255, 255, 255);
        $this->pdf->SetXY(20, 12);
        $candidate_name = trim($app['first_name'] . ' ' . $app['last_name']);
        $this->pdf->Cell(120, 8, $candidate_name, 0, 1, 'L');
        
        // Professional title
        $this->pdf->SetFont('Arial', '', 11);
        $this->pdf->SetTextColor(255, 255, 255);
        $this->pdf->SetXY(20, 25);
        $this->pdf->Cell(120, 6, 'Applying for: ' . $app['job_title'], 0, 1, 'L');
        
        $this->pdf->Ln(20);
    }
    
    private function drawModernProfilePlaceholder($x, $y, $size) {
        // Modern rectangular placeholder
        $this->pdf->SetFillColor(255, 255, 255);
        $this->pdf->Rect($x, $y, $size, $size, 'F');
        $this->pdf->SetDrawColor($this->lightAccent[0], $this->lightAccent[1], $this->lightAccent[2]);
        $this->pdf->Rect($x, $y, $size, $size, 'D');
        
        // Add "Photo" text
        $this->pdf->SetFont('Arial', '', 8);
        $this->pdf->SetTextColor($this->mediumGray[0], $this->mediumGray[1], $this->mediumGray[2]);
        $this->pdf->SetXY($x, $y + $size/2 - 2);
        $this->pdf->Cell($size, 4, 'Photo', 0, 0, 'C');
    }
    
    private function addProfileIntroduction($app, $profile_data) {
        // Profile Introduction Section
        $this->pdf->SetFillColor($this->lightGray[0], $this->lightGray[1], $this->lightGray[2]);
        $this->pdf->Rect(20, $this->pdf->GetY(), 170, 25, 'F');
        
        $start_y = $this->pdf->GetY();
        
        // Section title
        $this->pdf->SetFont('Arial', 'B', 12);
        $this->pdf->SetTextColor($this->primaryColor[0], $this->primaryColor[1], $this->primaryColor[2]);
        $this->pdf->SetXY(25, $start_y + 3);
        $this->pdf->Cell(160, 6, 'PROFILE INTRODUCTION', 0, 1, 'L');
        
        // Profile text
        $this->pdf->SetFont('Arial', '', 10);
        $this->pdf->SetTextColor($this->textColor[0], $this->textColor[1], $this->textColor[2]);
        $this->pdf->SetXY(25, $start_y + 10);
        
        // Generate dynamic profile introduction
        $intro = $this->generateProfileIntro($app, $profile_data);
        $this->pdf->MultiCell(160, 4, $intro, 0, 'L');
        
        $this->pdf->SetY($start_y + 30);
    }
    
    private function generateProfileIntro($app, $profile_data) {
        $name = $app['first_name'];
        $type = !empty($profile_data['candidate_type']) ? strtolower($profile_data['candidate_type']) : 'external';
        $nationality = $this->determineNationality($profile_data);
        
        $intro = "$name is an " . ($type === 'internal' ? 'internal' : 'external') . " candidate";
        
        if ($nationality) {
            $intro .= " of $nationality nationality";
        }
        
        $intro .= " applying for the position of {$app['job_title']}. ";
        
        if ($type === 'internal') {
            $intro .= "As an existing employee, $name brings valuable institutional knowledge and understanding of company culture and processes.";
        } else {
            $intro .= "$name brings external expertise and fresh perspectives to contribute to the organization's growth and success.";
        }
        
        return $intro;
    }
    
    private function determineNationality($profile_data) {
        // Check both sa_id_number and id_number fields for South African ID
        if (!empty($profile_data['sa_id_number']) || !empty($profile_data['id_number'])) {
            return 'South African';
        }
        
        // Check if marked as foreign national
        if (!empty($profile_data['is_foreign_national']) && $profile_data['is_foreign_national'] === 'yes') {
            // If has nationality specified, use it
            if (!empty($profile_data['nationality'])) {
                return $profile_data['nationality'];
            }
            // If has country specified, use it
            if (!empty($profile_data['country'])) {
                return $profile_data['country'];
            }
            return 'Foreign National';
        }
        
        // If has passport but no SA ID, check if nationality is specified
        if (!empty($profile_data['passport_number'])) {
            return !empty($profile_data['nationality']) ? $profile_data['nationality'] : 'Foreign';
        }
        
        // Default to specified nationality or South African if nothing else
        if (!empty($profile_data['nationality'])) {
            return $profile_data['nationality'];
        }
        
        // If is_foreign_national is 'no' or not set, assume South African
        return 'South African';
    }
    
    private function addApplicationSummary($app, $profile_data) {
        // Modern Application Summary Box
        $this->pdf->SetFillColor($this->lightGray[0], $this->lightGray[1], $this->lightGray[2]);
        $this->pdf->Rect(20, $this->pdf->GetY(), 170, 45, 'F');
        
        $this->pdf->SetDrawColor($this->accentColor[0], $this->accentColor[1], $this->accentColor[2]);
        $this->pdf->Rect(20, $this->pdf->GetY(), 170, 45, 'D');
        
        $start_y = $this->pdf->GetY();
        
        // Title
        $this->pdf->SetFont('Arial', 'B', 12);
        $this->pdf->SetTextColor($this->primaryColor[0], $this->primaryColor[1], $this->primaryColor[2]);
        $this->pdf->SetXY(25, $start_y + 5);
        $this->pdf->Cell(160, 6, 'APPLICATION SUMMARY', 0, 1, 'L');
        
        // Position Applied For - On its own row
        $this->pdf->SetFont('Arial', 'B', 11);
        $this->pdf->SetTextColor($this->textColor[0], $this->textColor[1], $this->textColor[2]);
        $this->pdf->SetXY(25, $start_y + 15);
        $this->pdf->Cell(160, 5, 'Position Applied For:', 0, 1, 'L');
        
        $this->pdf->SetFont('Arial', '', 10);
        $this->pdf->SetXY(25, $start_y + 21);
        $this->pdf->Cell(160, 5, $app['job_title'], 0, 1, 'L');
        
        // Other details in two columns
        $this->pdf->SetFont('Arial', '', 10);
        $col1_x = 25;
        $col2_x = 110;
        $detail_y = $start_y + 28;
        
        // Column 1
        $this->pdf->SetXY($col1_x, $detail_y);
        $this->pdf->Cell(80, 5, 'Application Date: ' . date('d M Y', strtotime($app['applied_at'])), 0, 1, 'L');
        
        $this->pdf->SetX($col1_x);
        $this->pdf->Cell(80, 5, 'Status: ' . ucfirst($app['status']), 0, 1, 'L');
        
        // ID Number - check both fields
        $id_number = $profile_data['sa_id_number'] ?? $profile_data['id_number'] ?? '';
        if (!empty($id_number)) {
            $this->pdf->SetX($col1_x);
            $this->pdf->Cell(80, 5, 'ID Number: ' . $id_number, 0, 1, 'L');
        }
        
        // Column 2
        $this->pdf->SetXY($col2_x, $detail_y);
        
        // Nationality
        $nationality = $this->determineNationality($profile_data);
        if (empty($nationality)) $nationality = 'Not specified';
        $this->pdf->Cell(75, 5, 'Nationality: ' . $nationality, 0, 1, 'L');
        
        // Candidate Type (Internal/External)
        $this->pdf->SetX($col2_x);
        $candidate_type = !empty($profile_data['candidate_type']) ? ucfirst($profile_data['candidate_type']) : 'External';
        $this->pdf->Cell(75, 5, 'Candidate Type: ' . $candidate_type, 0, 1, 'L');
        
        // Score if available
        $this->pdf->SetX($col2_x);
        if (!empty($app['total_score']) && $app['total_score'] > 0) {
            $this->pdf->Cell(75, 5, 'Score: ' . number_format($app['total_score'], 1) . '%', 0, 1, 'L');
        } else {
            $this->pdf->Cell(75, 5, 'Score: Not scored', 0, 1, 'L');
        }
        
        // Move past the summary box
        $this->pdf->SetY($start_y + 50);
    }
    
    private function addContactInfo($app, $profile_data) {
        $this->addSectionTitle('COMPLETE PROFILE INFORMATION');
        
        $this->pdf->SetFont('Arial', 'B', 11);
        $this->pdf->SetTextColor($this->accentColor[0], $this->accentColor[1], $this->accentColor[2]);
        
        // Contact Information Column
        $col1_x = 20;
        $col2_x = 85;
        $col3_x = 150;
        $start_y = $this->pdf->GetY();
        
        // Track the maximum height needed for all columns
        $max_height = 0;
        
        // Column 1: Contact Information
        $this->pdf->SetXY($col1_x, $start_y);
        $this->pdf->Cell(60, 6, 'Contact Information', 0, 1, 'L');
        $this->pdf->SetFont('Arial', '', 9);
        $this->pdf->SetTextColor($this->textColor[0], $this->textColor[1], $this->textColor[2]);
        
        $col1_y = $this->pdf->GetY();
        
        $this->pdf->SetX($col1_x);
        $this->pdf->Cell(60, 5, 'Email: ' . $app['email'], 0, 1, 'L');
        
        if (!empty($profile_data['phone'])) {
            $this->pdf->SetX($col1_x);
            $this->pdf->Cell(60, 5, 'Phone: ' . $profile_data['phone'], 0, 1, 'L');
        }
        
        if (!empty($profile_data['address'])) {
            $this->pdf->SetX($col1_x);
            // Split long addresses into multiple lines
            $address_lines = explode("\n", wordwrap($profile_data['address'], 35, "\n"));
            foreach ($address_lines as $i => $line) {
                $prefix = ($i === 0) ? 'Address: ' : '         ';
                $this->pdf->SetX($col1_x);
                $this->pdf->Cell(60, 5, $prefix . $line, 0, 1, 'L');
            }
        }
        
        $col1_end_y = $this->pdf->GetY();
        $max_height = max($max_height, $col1_end_y - $start_y);
        
        // Column 2: Personal Details
        $this->pdf->SetXY($col2_x, $start_y);
        $this->pdf->SetFont('Arial', 'B', 11);
        $this->pdf->SetTextColor($this->accentColor[0], $this->accentColor[1], $this->accentColor[2]);
        $this->pdf->Cell(60, 6, 'Personal Details', 0, 1, 'L');
        
        $this->pdf->SetFont('Arial', '', 9);
        $this->pdf->SetTextColor($this->textColor[0], $this->textColor[1], $this->textColor[2]);
        
        if (!empty($profile_data['date_of_birth'])) {
            $this->pdf->SetX($col2_x);
            $this->pdf->Cell(60, 5, 'Date of Birth: ' . $profile_data['date_of_birth'], 0, 1, 'L');
        }
        
        if (!empty($profile_data['gender'])) {
            $this->pdf->SetX($col2_x);
            $this->pdf->Cell(60, 5, 'Gender: ' . ucfirst($profile_data['gender']), 0, 1, 'L');
        }
        
        $nationality = $this->determineNationality($profile_data);
        if (!empty($nationality)) {
            $this->pdf->SetX($col2_x);
            $this->pdf->Cell(60, 5, 'Nationality: ' . $nationality, 0, 1, 'L');
        }
        
        if (!empty($profile_data['candidate_type'])) {
            $this->pdf->SetX($col2_x);
            $candidate_type = ucfirst($profile_data['candidate_type']);
            $this->pdf->Cell(60, 5, 'Type: ' . $candidate_type . ' Candidate', 0, 1, 'L');
        }
        
        $id_number = $profile_data['sa_id_number'] ?? $profile_data['id_number'] ?? '';
        if (!empty($id_number)) {
            $this->pdf->SetX($col2_x);
            $this->pdf->Cell(60, 5, 'ID Number: ' . $id_number, 0, 1, 'L');
        }
        
        $col2_end_y = $this->pdf->GetY();
        $max_height = max($max_height, $col2_end_y - $start_y);
        
        // Set Y position to the bottom of the tallest column plus some spacing
        $this->pdf->SetY($start_y + $max_height + 10);
    }
    
    private function addEducationSection($cv_data) {
        if (empty($cv_data['education'])) return;
        
        $this->addSectionTitle('EDUCATION');
        
        foreach ($cv_data['education'] as $edu) {
            // Check if we need a new page
            if ($this->pdf->GetY() > 250) {
                $this->pdf->AddPage();
            }
            
            $this->pdf->SetFont('Arial', 'B', 11);
            $this->pdf->SetTextColor($this->textColor[0], $this->textColor[1], $this->textColor[2]);
            
            $degree = $edu['degree'] ?? 'Degree';
            $institution = $edu['institution'] ?? 'Institution';
            $dates = ($edu['start_date'] ?? '') . ' - ' . ($edu['end_date'] ?? 'Present');
            
            $this->pdf->Cell(130, 6, $degree, 0, 0, 'L');
            $this->pdf->SetFont('Arial', '', 9);
            $this->pdf->Cell(50, 6, $dates, 0, 1, 'R');
            
            $this->pdf->SetFont('Arial', 'I', 10);
            $this->pdf->SetTextColor(100, 100, 100);
            $this->pdf->Cell(180, 5, $institution, 0, 1, 'L');
            
            if (!empty($edu['description'])) {
                $this->pdf->SetFont('Arial', '', 9);
                $this->pdf->SetTextColor($this->textColor[0], $this->textColor[1], $this->textColor[2]);
                $this->pdf->MultiCell(180, 5, $edu['description'], 0, 'L');
            }
            
            $this->pdf->Ln(5);
        }
    }
    
    private function addCertificationsSection($qualifications) {
        if (empty($qualifications['certificates'])) return;
        
        $this->addSectionTitle('CERTIFICATIONS & LICENSES');
        
        foreach ($qualifications['certificates'] as $cert) {
            $this->pdf->SetFont('Arial', 'B', 11);
            $this->pdf->SetTextColor($this->textColor[0], $this->textColor[1], $this->textColor[2]);
            
            $title = $cert['title'] ?? 'Certificate';
            $organization = $cert['issuing_organization'] ?? '';
            $issue_date = $cert['issue_date'] ?? '';
            
            $this->pdf->Cell(130, 6, $title, 0, 0, 'L');
            $this->pdf->SetFont('Arial', '', 9);
            $this->pdf->Cell(50, 6, 'Issued: ' . $issue_date, 0, 1, 'R');
            
            if ($organization) {
                $this->pdf->SetFont('Arial', 'I', 10);
                $this->pdf->SetTextColor(100, 100, 100);
                $this->pdf->Cell(180, 5, $organization, 0, 1, 'L');
            }
            
            if (!empty($cert['description'])) {
                $this->pdf->SetFont('Arial', '', 9);
                $this->pdf->SetTextColor($this->textColor[0], $this->textColor[1], $this->textColor[2]);
                $this->pdf->MultiCell(180, 5, $cert['description'], 0, 'L');
            }
            
            $this->pdf->Ln(3);
        }
        
        $this->pdf->Ln(5);
    }
    
    private function addExperienceSection($cv_data) {
        if (empty($cv_data['experience'])) return;
        
        $this->addSectionTitle('WORK EXPERIENCE');
        
        foreach ($cv_data['experience'] as $exp) {
            $this->pdf->SetFont('Arial', 'B', 11);
            $this->pdf->SetTextColor($this->textColor[0], $this->textColor[1], $this->textColor[2]);
            
            $position = $exp['position'] ?? 'Position';
            $company = $exp['company_name'] ?? 'Company';
            $dates = ($exp['start_date'] ?? '') . ' - ' . ($exp['end_date'] ?? 'Present');
            
            $this->pdf->Cell(130, 6, $position, 0, 0, 'L');
            $this->pdf->SetFont('Arial', '', 9);
            $this->pdf->Cell(50, 6, $dates, 0, 1, 'R');
            
            $this->pdf->SetFont('Arial', 'I', 10);
            $this->pdf->SetTextColor(100, 100, 100);
            $this->pdf->Cell(180, 5, $company, 0, 1, 'L');
            
            if (!empty($exp['description'])) {
                $this->pdf->SetFont('Arial', '', 9);
                $this->pdf->SetTextColor($this->textColor[0], $this->textColor[1], $this->textColor[2]);
                $this->pdf->MultiCell(180, 5, $exp['description'], 0, 'L');
            }
            
            $this->pdf->Ln(3);
        }
        
        $this->pdf->Ln(5);
    }
    
    private function addReferencesSection($cv_data) {
        if (empty($cv_data['references'])) return;
        
        $this->addSectionTitle('PROFESSIONAL REFERENCES');
        
        foreach ($cv_data['references'] as $ref) {
            $this->pdf->SetFont('Arial', 'B', 11);
            $this->pdf->SetTextColor($this->textColor[0], $this->textColor[1], $this->textColor[2]);
            
            $name = $ref['name'] ?? 'Reference';
            $position = $ref['position'] ?? '';
            $company = $ref['company'] ?? '';
            
            $this->pdf->Cell(180, 6, $name, 0, 1, 'L');
            
            if ($position || $company) {
                $this->pdf->SetFont('Arial', 'I', 10);
                $this->pdf->SetTextColor(100, 100, 100);
                $title_line = $position;
                if ($position && $company) $title_line .= ' | ';
                $title_line .= $company;
                $this->pdf->Cell(180, 5, $title_line, 0, 1, 'L');
            }
            
            $this->pdf->SetFont('Arial', '', 9);
            $this->pdf->SetTextColor($this->textColor[0], $this->textColor[1], $this->textColor[2]);
            
            if (!empty($ref['email'])) {
                $this->pdf->Cell(90, 5, 'Email: ' . $ref['email'], 0, 0, 'L');
            }
            if (!empty($ref['phone'])) {
                $this->pdf->Cell(90, 5, 'Phone: ' . $ref['phone'], 0, 1, 'L');
            } else {
                $this->pdf->Ln(5);
            }
            
            $this->pdf->Ln(3);
        }
        
        $this->pdf->Ln(5);
    }
    
    private function addSkillsSection($cv_data) {
        // Check for skills from CV data or generate from experience/education
        $skills = [];
        
        if (!empty($cv_data['skills'])) {
            foreach ($cv_data['skills'] as $skill) {
                $skills[] = $skill['skill_name'] ?? $skill['name'] ?? '';
            }
        }
        
        // If no skills found, generate some from experience descriptions
        if (empty($skills) && !empty($cv_data['experience'])) {
            $skills = $this->extractSkillsFromExperience($cv_data['experience']);
        }
        
        // Add some default professional skills if still empty
        if (empty($skills)) {
            $skills = ['Communication', 'Problem Solving', 'Team Collaboration', 'Time Management', 'Microsoft Office'];
        }
        
        if (empty($skills)) return;
        
        $this->addModernSectionTitle('CORE SKILLS & COMPETENCIES');
        
        // Modern skills grid layout
        $this->pdf->SetFont('Arial', '', 9);
        $this->pdf->SetTextColor(255, 255, 255);
        
        $x = 20;
        $y = $this->pdf->GetY();
        $skill_width = 52;
        $skill_height = 8;
        $cols = 3;
        $row = 0;
        $col = 0;
        
        foreach ($skills as $index => $skill) {
            if (empty($skill)) continue;
            
            $skill_x = $x + ($col * ($skill_width + 5));
            $skill_y = $y + ($row * ($skill_height + 3));
            
            // Skill box background
            $this->pdf->SetFillColor($this->accentColor[0], $this->accentColor[1], $this->accentColor[2]);
            $this->pdf->Rect($skill_x, $skill_y, $skill_width, $skill_height, 'F');
            
            // Skill text
            $this->pdf->SetXY($skill_x + 2, $skill_y + 1.5);
            $this->pdf->Cell($skill_width - 4, 5, $skill, 0, 0, 'C');
            
            $col++;
            if ($col >= $cols) {
                $col = 0;
                $row++;
            }
        }
        
        // Move Y position past skills grid
        $total_rows = ceil(count($skills) / $cols);
        $this->pdf->SetY($y + ($total_rows * ($skill_height + 3)) + 10);
    }
    
    private function extractSkillsFromExperience($experience) {
        $skills = [];
        $common_skills = [
            'management', 'leadership', 'analysis', 'planning', 'communication',
            'problem solving', 'team work', 'project management', 'training',
            'customer service', 'sales', 'marketing', 'finance', 'accounting'
        ];
        
        foreach ($experience as $exp) {
            $description = strtolower($exp['description'] ?? '');
            foreach ($common_skills as $skill) {
                if (strpos($description, $skill) !== false && !in_array(ucwords($skill), $skills)) {
                    $skills[] = ucwords($skill);
                    if (count($skills) >= 6) break 2; // Limit to 6 skills
                }
            }
        }
        
        return $skills;
    }
    
    private function addSectionTitle($title) {
        // Add some space before section title
        $this->pdf->Ln(5);
        
        $this->pdf->SetFont('Arial', 'B', 12);
        $this->pdf->SetTextColor($this->accentColor[0], $this->accentColor[1], $this->accentColor[2]);
        $this->pdf->Cell(180, 8, $title, 0, 1, 'L');
        
        // Add underline
        $this->pdf->SetDrawColor($this->accentColor[0], $this->accentColor[1], $this->accentColor[2]);
        $this->pdf->Line(20, $this->pdf->GetY(), 190, $this->pdf->GetY());
        
        $this->pdf->Ln(8);
    }
    
    private function addModernSectionTitle($title) {
        // Modern section title with accent bar
        $this->pdf->Ln(3);
        
        // Left accent bar
        $this->pdf->SetFillColor($this->accentColor[0], $this->accentColor[1], $this->accentColor[2]);
        $this->pdf->Rect(20, $this->pdf->GetY(), 4, 8, 'F');
        
        $this->pdf->SetFont('Arial', 'B', 12);
        $this->pdf->SetTextColor($this->primaryColor[0], $this->primaryColor[1], $this->primaryColor[2]);
        $this->pdf->SetXY(28, $this->pdf->GetY());
        $this->pdf->Cell(160, 8, $title, 0, 1, 'L');
        
        $this->pdf->Ln(5);
    }
}
?>