<?php
require_once('fpdf.php');

class FPDFReportGenerator extends FPDF {
    private $reportData;
    private $reportTitle;
    private $period;
    private $companyName;
    
    public function __construct($reportData, $reportTitle, $period, $companyName) {
        parent::__construct();
        $this->reportData = $reportData;
        $this->reportTitle = $reportTitle;
        $this->period = $period;
        $this->companyName = $companyName;
    }
    
    // Page header
    function Header() {
        $this->SetFont('Arial', 'B', 20);
        $this->SetTextColor(102, 126, 234);
        $this->Cell(0, 15, $this->reportTitle, 0, 1, 'C');
        $this->Ln(5);
        
        $this->SetFont('Arial', '', 14);
        $this->SetTextColor(0, 0, 0);
        $this->Cell(0, 8, $this->companyName, 0, 1, 'C');
        $this->Cell(0, 8, 'Generated on ' . date('F j, Y \a\t g:i A'), 0, 1, 'C');
        $this->Cell(0, 8, 'Period: ' . $this->period, 0, 1, 'C');
        $this->Ln(10);
        
        // Draw a line
        $this->SetDrawColor(102, 126, 234);
        $this->SetLineWidth(0.5);
        $this->Line(20, $this->GetY(), 190, $this->GetY());
        $this->Ln(10);
    }
    
    // Page footer
    function Footer() {
        $this->SetY(-30);
        $this->SetFont('Arial', 'I', 8);
        $this->SetTextColor(128, 128, 128);
        $this->Cell(0, 10, 'Report generated by HR Analytics System', 0, 1, 'C');
        $this->Cell(0, 10, 'Page ' . $this->PageNo() . '/{nb}', 0, 0, 'C');
    }
    
    public function generateReport() {
        $this->AliasNbPages();
        $this->AddPage();
        $this->SetFont('Arial', '', 12);
        
        // Add recruitment metrics
        if (isset($this->reportData['metrics'])) {
            $this->addRecruitmentSummary();
        }
        
        // Add conversion rates
        if (isset($this->reportData['conversion'])) {
            $this->addConversionRates();
        }
        
        // Add hiring funnel
        if (isset($this->reportData['funnel']) && !empty($this->reportData['funnel'])) {
            $this->addHiringFunnel();
        }
        
        // Add source effectiveness
        if (isset($this->reportData['sources']) && !empty($this->reportData['sources'])) {
            $this->addSourceEffectiveness();
        }
        
        // Add time to hire stats
        if (isset($this->reportData['timeToHire'])) {
            $this->addTimeToHireStats();
        }
        
        return $this->Output('S'); // Return PDF as string
    }
    
    private function addRecruitmentSummary() {
        $this->SetFont('Arial', 'B', 16);
        $this->SetTextColor(102, 126, 234);
        $this->Cell(0, 10, 'Recruitment Summary', 0, 1);
        $this->Ln(5);
        
        $metrics = $this->reportData['metrics'];
        
        // Create a 2x2 grid for metrics
        $this->SetFont('Arial', '', 12);
        $this->SetTextColor(0, 0, 0);
        
        $cellWidth = 85;
        $cellHeight = 25;
        
        // Row 1
        $this->SetFillColor(240, 248, 255);
        $this->Cell($cellWidth, $cellHeight, 'Applications Received', 1, 0, 'C', true);
        $this->SetFont('Arial', 'B', 16);
        $this->Cell($cellWidth, $cellHeight, $metrics['application_received']['total'] ?? 0, 1, 1, 'C', true);
        
        $this->SetFont('Arial', '', 12);
        $this->Cell($cellWidth, $cellHeight, 'Shortlisted', 1, 0, 'C', true);
        $this->SetFont('Arial', 'B', 16);
        $this->Cell($cellWidth, $cellHeight, $metrics['shortlisted']['total'] ?? 0, 1, 1, 'C', true);
        
        $this->SetFont('Arial', '', 12);
        $this->Cell($cellWidth, $cellHeight, 'Interviewed', 1, 0, 'C', true);
        $this->SetFont('Arial', 'B', 16);
        $this->Cell($cellWidth, $cellHeight, $metrics['interviewed']['total'] ?? 0, 1, 1, 'C', true);
        
        $this->SetFont('Arial', '', 12);
        $this->Cell($cellWidth, $cellHeight, 'Hired', 1, 0, 'C', true);
        $this->SetFont('Arial', 'B', 16);
        $this->Cell($cellWidth, $cellHeight, $metrics['hired']['total'] ?? 0, 1, 1, 'C', true);
        
        $this->Ln(10);
    }
    
    private function addConversionRates() {
        $this->SetFont('Arial', 'B', 16);
        $this->SetTextColor(102, 126, 234);
        $this->Cell(0, 10, 'Conversion Rates', 0, 1);
        $this->Ln(5);
        
        $conversion = $this->reportData['conversion'];
        
        $this->SetFont('Arial', '', 12);
        $this->SetTextColor(0, 0, 0);
        $this->SetFillColor(248, 249, 250);
        
        // Table header
        $this->Cell(120, 10, 'Conversion Stage', 1, 0, 'C', true);
        $this->Cell(50, 10, 'Rate', 1, 1, 'C', true);
        
        // Table rows
        $conversions = [
            'Application → Shortlist' => $conversion['application_to_shortlist'] ?? 0,
            'Shortlist → Interview' => $conversion['shortlist_to_interview'] ?? 0,
            'Interview → Hire' => $conversion['interview_to_hire'] ?? 0,
            'Overall Conversion' => $conversion['application_to_hire'] ?? 0
        ];
        
        foreach ($conversions as $stage => $rate) {
            $this->Cell(120, 10, $stage, 1, 0, 'L');
            $this->SetFont('Arial', 'B', 12);
            $this->Cell(50, 10, $rate . '%', 1, 1, 'C');
            $this->SetFont('Arial', '', 12);
        }
        
        $this->Ln(10);
    }
    
    private function addHiringFunnel() {
        $this->SetFont('Arial', 'B', 16);
        $this->SetTextColor(102, 126, 234);
        $this->Cell(0, 10, 'Hiring Funnel Analysis', 0, 1);
        $this->Ln(5);
        
        $this->SetFont('Arial', '', 10);
        $this->SetTextColor(0, 0, 0);
        $this->SetFillColor(248, 249, 250);
        
        // Table header
        $this->Cell(50, 10, 'Job Position', 1, 0, 'C', true);
        $this->Cell(25, 10, 'Applications', 1, 0, 'C', true);
        $this->Cell(25, 10, 'Shortlisted', 1, 0, 'C', true);
        $this->Cell(25, 10, 'Interviewed', 1, 0, 'C', true);
        $this->Cell(20, 10, 'Hired', 1, 0, 'C', true);
        $this->Cell(25, 10, 'Success %', 1, 1, 'C', true);
        
        // Table rows
        foreach ($this->reportData['funnel'] as $job) {
            $successRate = $job['applications'] > 0 ? round(($job['hired'] / $job['applications']) * 100, 1) : 0;
            
            $this->Cell(50, 8, substr($job['job_title'], 0, 20), 1, 0, 'L');
            $this->Cell(25, 8, $job['applications'], 1, 0, 'C');
            $this->Cell(25, 8, $job['shortlisted'] ?? 0, 1, 0, 'C');
            $this->Cell(25, 8, $job['interviewed'] ?? 0, 1, 0, 'C');
            $this->Cell(20, 8, $job['hired'], 1, 0, 'C');
            $this->Cell(25, 8, $successRate . '%', 1, 1, 'C');
        }
        
        $this->Ln(10);
    }
    
    private function addSourceEffectiveness() {
        $this->SetFont('Arial', 'B', 16);
        $this->SetTextColor(102, 126, 234);
        $this->Cell(0, 10, 'Source Effectiveness Analysis', 0, 1);
        $this->Ln(5);
        
        $this->SetFont('Arial', '', 12);
        $this->SetTextColor(0, 0, 0);
        $this->SetFillColor(248, 249, 250);
        
        // Table header
        $this->Cell(40, 10, 'Source', 1, 0, 'C', true);
        $this->Cell(30, 10, 'Applications', 1, 0, 'C', true);
        $this->Cell(30, 10, 'Hires', 1, 0, 'C', true);
        $this->Cell(35, 10, 'Conversion %', 1, 0, 'C', true);
        $this->Cell(35, 10, 'ROI Rating', 1, 1, 'C', true);
        
        // Table rows
        foreach ($this->reportData['sources'] as $source) {
            $rating = $source['conversion_rate'] > 15 ? 'Excellent' : 
                     ($source['conversion_rate'] > 10 ? 'Good' : 
                     ($source['conversion_rate'] > 5 ? 'Average' : 'Poor'));
            
            $this->Cell(40, 8, ucfirst($source['source_name']), 1, 0, 'L');
            $this->Cell(30, 8, $source['applications_count'], 1, 0, 'C');
            $this->Cell(30, 8, $source['hires_count'], 1, 0, 'C');
            $this->Cell(35, 8, $source['conversion_rate'] . '%', 1, 0, 'C');
            $this->Cell(35, 8, $rating, 1, 1, 'C');
        }
        
        $this->Ln(10);
    }
    
    private function addTimeToHireStats() {
        $this->SetFont('Arial', 'B', 16);
        $this->SetTextColor(102, 126, 234);
        $this->Cell(0, 10, 'Time to Hire Statistics', 0, 1);
        $this->Ln(5);
        
        $timeStats = $this->reportData['timeToHire'];
        
        $this->SetFont('Arial', '', 12);
        $this->SetTextColor(0, 0, 0);
        $this->SetFillColor(240, 248, 255);
        
        $cellWidth = 56.67;
        $cellHeight = 25;
        
        // Three metrics in a row
        $this->Cell($cellWidth, $cellHeight, 'Average Time to Hire', 1, 0, 'C', true);
        $this->Cell($cellWidth, $cellHeight, 'Fastest Hire', 1, 0, 'C', true);
        $this->Cell($cellWidth, $cellHeight, 'Longest Process', 1, 1, 'C', true);
        
        $this->SetFont('Arial', 'B', 16);
        $this->Cell($cellWidth, $cellHeight, round($timeStats['avg_time_to_hire'] ?? 0) . ' days', 1, 0, 'C', true);
        $this->Cell($cellWidth, $cellHeight, ($timeStats['min_time_to_hire'] ?? 0) . ' days', 1, 0, 'C', true);
        $this->Cell($cellWidth, $cellHeight, ($timeStats['max_time_to_hire'] ?? 0) . ' days', 1, 1, 'C', true);
        
        $this->Ln(10);
    }
}
?>