<?php
require_once 'config/config.php';

$database = new Database();
$db = $database->getConnection();

echo "<h2>🗓️ Check Current Deadline Usage</h2>\n";

// 1. Check current deadline values
echo "<h3>1. Current Deadline Data</h3>\n";
try {
    $stmt = $db->query("SELECT id, title, deadline, status, created_at FROM jobs ORDER BY deadline DESC, created_at DESC");
    $jobs = $stmt->fetchAll();
    
    echo "<table border='1' cellpadding='5'>\n";
    echo "<tr><th>ID</th><th>Title</th><th>Deadline</th><th>Status</th><th>Days Until Deadline</th></tr>\n";
    
    $today = new DateTime();
    
    foreach ($jobs as $job) {
        echo "<tr>";
        echo "<td>{$job['id']}</td>";
        echo "<td>" . substr($job['title'], 0, 40) . "...</td>";
        
        if ($job['deadline']) {
            $deadline = new DateTime($job['deadline']);
            $interval = $today->diff($deadline);
            $days_diff = $today <= $deadline ? $interval->days : -$interval->days;
            
            $deadline_color = '';
            if ($days_diff < 0) {
                $deadline_color = 'style="background-color: #ffebee; color: #d32f2f;"'; // Expired - red
            } elseif ($days_diff <= 7) {
                $deadline_color = 'style="background-color: #fff3cd; color: #856404;"'; // Closing soon - yellow
            } else {
                $deadline_color = 'style="background-color: #e8f5e8; color: #2e7d32;"'; // Good - green
            }
            
            echo "<td $deadline_color>{$job['deadline']}</td>";
            echo "<td>{$job['status']}</td>";
            
            if ($days_diff < 0) {
                echo "<td style='color: #d32f2f; font-weight: bold;'>❌ EXPIRED ({$interval->days} days ago)</td>";
            } elseif ($days_diff == 0) {
                echo "<td style='color: #ff9800; font-weight: bold;'>⚠️ TODAY</td>";
            } elseif ($days_diff <= 7) {
                echo "<td style='color: #ff9800; font-weight: bold;'>⚠️ {$days_diff} days left</td>";
            } else {
                echo "<td style='color: #2e7d32;'>✅ {$days_diff} days left</td>";
            }
        } else {
            echo "<td style='color: #9e9e9e;'>No deadline</td>";
            echo "<td>{$job['status']}</td>";
            echo "<td style='color: #9e9e9e;'>N/A</td>";
        }
        
        echo "</tr>\n";
    }
    echo "</table>\n";
    
} catch (Exception $e) {
    echo "<p>❌ Error getting deadline data: " . $e->getMessage() . "</p>\n";
}

// 2. Check where job filtering currently happens
echo "<h3>2. Current Filtering Analysis</h3>\n";

$files_to_check = [
    'jobs.php',
    'index.php', 
    'job-details.php'
];

foreach ($files_to_check as $file) {
    if (file_exists($file)) {
        $content = file_get_contents($file);
        $has_deadline_check = stripos($content, 'deadline') !== false;
        $has_date_filter = preg_match('/WHERE.*date.*<|WHERE.*deadline.*<|WHERE.*NOW\(\)|WHERE.*CURDATE\(\)/i', $content);
        
        echo "<div style='background: #f8f9fa; padding: 10px; margin: 5px 0; border-radius: 5px;'>\n";
        echo "<h4>📄 $file</h4>\n";
        echo "<p><strong>Mentions deadline:</strong> " . ($has_deadline_check ? "✅ Yes" : "❌ No") . "</p>\n";
        echo "<p><strong>Has date filtering:</strong> " . ($has_date_filter ? "✅ Yes" : "❌ No") . "</p>\n";
        echo "</div>\n";
    }
}

// 3. Check application process
echo "<h3>3. Application Process Analysis</h3>\n";

// Find how applications are processed
if (file_exists('job-details.php')) {
    $content = file_get_contents('job-details.php');
    
    // Look for application form or apply button
    if (preg_match('/apply|submit.*application/i', $content)) {
        echo "<p>✅ Found application functionality in job-details.php</p>\n";
        
        // Check if there's deadline validation
        $has_deadline_validation = stripos($content, 'deadline') !== false;
        echo "<p><strong>Has deadline validation:</strong> " . ($has_deadline_validation ? "✅ Yes" : "❌ No") . "</p>\n";
    } else {
        echo "<p>❌ No application functionality found in job-details.php</p>\n";
    }
}

// 4. Statistics
echo "<h3>4. Deadline Statistics</h3>\n";
try {
    $stmt = $db->query("
        SELECT 
            CASE 
                WHEN deadline IS NULL THEN 'No Deadline'
                WHEN deadline < CURDATE() THEN 'Expired'
                WHEN deadline = CURDATE() THEN 'Expires Today'
                WHEN deadline <= DATE_ADD(CURDATE(), INTERVAL 7 DAY) THEN 'Expires Soon (7 days)'
                ELSE 'Active'
            END as deadline_status,
            COUNT(*) as count
        FROM jobs 
        WHERE status = 'active'
        GROUP BY deadline_status
    ");
    $stats = $stmt->fetchAll();
    
    echo "<table border='1' cellpadding='5'>\n";
    echo "<tr><th>Deadline Status</th><th>Count</th><th>Action Needed</th></tr>\n";
    
    foreach ($stats as $stat) {
        $action = '';
        $row_color = '';
        
        switch ($stat['deadline_status']) {
            case 'Expired':
                $action = '❌ Block applications';
                $row_color = 'style="background-color: #ffebee;"';
                break;
            case 'Expires Today':
                $action = '⚠️ Last chance to apply';
                $row_color = 'style="background-color: #fff3cd;"';
                break;
            case 'Expires Soon (7 days)':
                $action = '⚠️ Show urgency';
                $row_color = 'style="background-color: #fff3cd;"';
                break;
            case 'Active':
                $action = '✅ Allow applications';
                $row_color = 'style="background-color: #e8f5e8;"';
                break;
            case 'No Deadline':
                $action = '📅 Consider adding deadline';
                $row_color = 'style="background-color: #f0f0f0;"';
                break;
        }
        
        echo "<tr $row_color>";
        echo "<td>{$stat['deadline_status']}</td>";
        echo "<td>{$stat['count']}</td>";
        echo "<td>$action</td>";
        echo "</tr>\n";
    }
    echo "</table>\n";
    
} catch (Exception $e) {
    echo "<p>❌ Error getting statistics: " . $e->getMessage() . "</p>\n";
}

// 5. Recommendations
echo "<h3>5. Implementation Recommendations</h3>\n";
echo "<div style='background: #e3f2fd; color: #1565c0; padding: 15px; border-radius: 5px; margin: 10px 0;'>\n";
echo "<h4>🎯 Priority Actions:</h4>\n";
echo "<ol>\n";
echo "<li><strong>Update job listings</strong> - Filter out jobs where <code>deadline < CURDATE()</code></li>\n";
echo "<li><strong>Update job-details.php</strong> - Hide apply button for expired jobs</li>\n";
echo "<li><strong>Add deadline validation</strong> - Check deadline before processing applications</li>\n";
echo "<li><strong>Update UI</strong> - Show deadline info and urgency indicators</li>\n";
echo "<li><strong>Auto-close jobs</strong> - Change status to 'closed' for expired jobs</li>\n";
echo "</ol>\n";
echo "</div>\n";

?>

<style>
body { font-family: Arial, sans-serif; margin: 20px; line-height: 1.6; }
table { width: 100%; margin: 20px 0; border-collapse: collapse; }
th, td { padding: 8px; text-align: left; border: 1px solid #ddd; }
th { background-color: #f2f2f2; }
h1, h2, h3, h4 { color: #333; }
</style>